<?php

/**
 * Blended Article Functions
 * @since 1.0
 */

/**
 * Retrieve list of latest posts or posts matching criteria.
 *
 * @param array $args
 * @return posts.
 */
function blended_articles($args = null) {


    $defaults = array(
        'posts_per_page' => 4,
        'category' => 0, 'orderby' => 'date',
        'order' => 'DESC', 'include' => array(),
        'exclude' => array(), 'meta_key' => '',
        'meta_value' => '', 'post_type' => 'post',
        'category_taxonomy' => "category", "tag_taxonomy" => "post_tag",
        'suppress_filters' => true
    );

    //Pagination
    $paged = (get_query_var('paged')) ? get_query_var('paged') : 1;
    $defaults['paged'] = $paged;

    $r = wp_parse_args($args, $defaults);

    $post_query = new WP_Query($r);

    if ($post_query->have_posts()):
        $articles = array();
        $post_data = array();
        $posts = $post_query->posts;
        foreach ($posts as $post):
            $post_data["title"] = $post->post_title;
            $post_data["content"] = apply_filters('the_content', $post->post_content);
            $post_data["excerpt"] = $post->post_excerpt;
            $post_data["slug"] = $post->post_name;
            $post_data["date"] = $post->post_date;
            $post_data["href"] = get_permalink($post->ID);
            $post_data["author"] = get_the_author_meta("nicename", $post->post_author);
            $post_data["modified_date"] = $post->post_modified;
            $post_data["status"] = $post->post_status;
            $post_data["tags"] = blended_article_terms($post->ID, $r["tag_taxonomy"]);
            $post_data["categories"] = blended_article_terms($post->ID, $r["category_taxonomy"]);

            //Set image object
            $post_data["image"] = blended_attachment_image(get_post_thumbnail_id($post->ID));

            $post_data["comment_count"] = $post->comment_count;

            //set articles
            $articles['items'][] = $post_data;
        endforeach;

        //Set pagination
        $articles['pagination'] = new Timber\Pagination(array(), $post_query);


        wp_reset_postdata();

        //We are done!
        return $articles;

    endif;

    return false;
}

/**
 * Retrieves post data given a post ID or post object.
 * @param int| $post
 * @return post|array
 */
function blended_article($post = null, $args = null) {

    $defaults = array(
        'category_taxonomy' => "category", "tag_taxonomy" => "post_tag",
    );

    $r = wp_parse_args($args, $defaults);

    $post = get_post($post);

    if ($post):
        $article = array();
        $article["title"] = $post->post_title;
        $article["content"] = apply_filters('the_content', $post->post_content);
        $article["excerpt"] = $post->post_excerpt;
        $article["slug"] = $post->post_name;
        $article["date"] = $post->post_date;
        $article["href"] = get_permalink($post->ID);
        $article["author"] = get_the_author_meta("nicename", $post->post_author);
        $article["modified_date"] = $post->post_modified;
        $article["status"] = $post->post_status;
        $article["tags"] = blended_article_terms($post->ID, $r["tag_taxonomy"]);
        $article["categories"] = blended_article_terms($post->ID, $r["category_taxonomy"]);

        //Set image object
        $article["image"] = blended_attachment_image(get_post_thumbnail_id($post->ID));

        $article["comment_count"] = $post->comment_count;

        //We are done!
        return $article;

    endif;

    return false;
}

/**
 * Get tags/categories of article
 * @param int $post_id
 * @param string $taxonomy
 * @return array|boolean
 */
function blended_article_terms($post_id, $taxonomy) {
    $post_id = absint($post_id);
    if (!$post_id) {
        return false;
    }

    if (empty($taxonomy))
        return false;
    $terms = get_the_terms($post_id, $taxonomy);

    $post_terms = array();

    if ($terms && !is_wp_error($terms)) :
        foreach ($terms as $term):
            $post_terms[] = array(
                "name" => $term->name,
                "slug" => $term->slug,
                "url" => get_term_link($term->term_id),
                "description" => $term->description
            );

        endforeach;
        return $post_terms;
    endif;
}

/**
 * Get attachment image
 * @param type $attachment_id
 * @return boolean|array
 */
function blended_attachment_image($attachment_id, $size = "full") {
    $attachment_id = absint($attachment_id);
    if (!$attachment_id) {
        return false;
    }

    //Get image object
    $attachment = new TimberImage($attachment_id);
    if ($attachment):
        $image = array(
            "id" => $attachment->ID,
            "title" => $attachment->post_title,
            'alt' => $attachment->alt(),
            'caption' => $attachment->post_excerpt,
            'description' => apply_filters('the_content', $attachment->post_content),
            'href' => get_permalink($attachment->ID),
            'src' => $attachment->src($size),
            'date' => $attachment->post_date,
            'author' => get_the_author_meta("nicename", $attachment->post_author),
            'modified_date' => $attachment->post_modified,
            'width' => $attachment->width,
            'height' => $attachment->height
        );

        //set file attribute to image information
        $image['file'] = array(
            'representation' => $attachment->file_loc,
            'mimetype' => $attachment->post_mime_type,
            'filename' => $attachment->file
        );

        //We are done!
        return $image;

    endif;
}

/**
 * Get terms formatted for given WP_Term object
 * @param object WP_Term Object
 * @param array
 * @return array
 */
function blended_format_categories($categories, $args = array()) {
    $data = array();
    if(!empty($categories)):
        foreach($categories as $category ):
            $image_id = get_term_meta($category->term_id, $args['image_meta'], true);
            $temp_data = array(
                "href" => get_term_link($category->term_id),
                "title" => $category->name,
                "slug" => $category->slug,
                "image" => blended_attachment_image($image_id, "gallery-list-thumb")
            );
            $data['items'][] = $temp_data;
        endforeach;
        return $data;
    endif;
    
}