<?php

namespace Blended\hostlib;

use Blended\Orm\DB_Backend;

class ItineraryRoute
{

    function getBackend($slug = null)
    {
       global $blendedConfig;
       $switch = $blendedConfig->get("setup_configuration", "BACKEND");
       $network    = new Network();
       switch ($switch) {
	 case "DB":
	    $backend = new DB_Backend($slug);
	    return $backend;
	 case "FS":
            $backend = new Backend($slug);
	    return $backend;
	 default:
            $backend = new Backend($slug);
	    return $backend;
	 }
    }

    /**
     * Add a admin user into account.
     *
     * @param string $slug       is package name.
     * @param string $sessionKey is host session key.
     */
    function post_addAccountAdmin($slug, $body, $sessionKey)
    {
        try {
            $backend     = $this->getBackend($slug);
            $network     = new Network();
            $controller  = new Controller($backend, $network);
            $controller->blended_session_setter();
            $controller->addAccountAdmin($slug, $body);
        }
        catch (\Exception $e) {
            throw $e;
        }       
    }

    /**
     * Retrieve current active theme from Host.
     *
     * @return string current active theme.
     */
    function get_active()
    {
        try {
            $backend     = $this->getBackend();
            $network     = new Network();
            $controller  = new Controller($backend, $network);
            $activeTheme = $controller->getActiveTheme();
        }
        catch (\Exception $e) {
            throw $e;
        }
        
        return $activeTheme;
    }
    
    /**
     * Set as current active theme to Host.
     *
     * @param  string $slug       is package name.
     * @param  string $sessionKey is host session key.
     * @return string current active theme.
     */
    function put_active($slug, $sessionKey)
    {
        try {
            $backend     = $this->getBackend($slug);
            $network     = new Network();
            $controller  = new Controller($backend, $network);
            $controller->blended_session_setter();
            $activeTheme = $controller->make_active($slug);
        }
        catch (\Exception $e) {
            throw $e;
        }
        
        return $activeTheme;
    }
    
    /**
     * Create package draft.
     *
     * @param  array   $body       is request body.
     * @param  string  $sessionKey is host session key.
     * @param  boolean $verbose    parameter to provide detail view of hyperlink objects.
     * @return array package jptf.
     */
    function post_createPackage($body, $sessionKey, $verbose)
    {
        try {
            $backend     = $this->getBackend($slug);
            $network     = new Network();
            $controller  = new Controller($backend, $network);
            $controller->blended_session_setter();
            $packageJptf = $controller->createPackage($body, $verbose);
        }
        catch (\Exception $e) {
            throw $e;
        }
        
        return $packageJptf;
    }
    
    /**
     * Update local package.
     *
     * @param  string $accountSlug is account slug.
     * @param  string $slug        is package name.
     * @param  array  $body        is request body.
     * @param  string $sessionKey  is host session key.
     * @return array package jptf.
     */
    function put_localUpdate($accountSlug, $slug, $body, $sessionKey)
    {
        try {
            $backend     = $this->getBackend($slug);
            $network     = new Network();
            $controller  = new Controller($backend, $network);
            $controller->blended_session_setter();
            $backend     = $this->getBackend($slug);
            $controller  = new Controller($backend, $network);
            if (isset($body['replaceList']) && !empty($body['replaceList']) && isset($body['previousPackageHash'])) {
                $packageJptf = $controller->localUpdate($accountSlug, $slug, $body['replaceList'], $body['previousPackageHash'], $body['uploadedImagesDict']);
            }
            if (isset($body['documents']) && isset($body['previousPackageHash']) && !empty($body['documents'])) {
                $packageJptf = $controller->localUpdatefiles($accountSlug, $slug, $body['documents'], $body['previousPackageHash']);
            } elseif(isset($body['updatePackageHash'])) {
                $packageJptf = $body['documents'];
                $packageIntermediary = $controller->deserialize_jptf($packageJptf);
                $packageSlug = $accountSlug . '/' . $slug;
                $controller->save_packageHash($packageSlug, $packageIntermediary);
            }
        }
        catch (\Exception $e) {
            throw $e;
        }
        
        return $packageJptf;
    }
    
    /**
     * Merge partial jptf to package object.
     *
     * @param  string $accountSlug is account slug.
     * @param  string $slug        is package name.
     * @param  array  $body        is request body.
     * @param  string $sessionKey  is host session key.
     * @return array package jptf.
     */
    function put_mergePackage($accountSlug, $slug, $body, $sessionKey)
    {
        try {
            $backend     = $this->getBackend($slug);
            $network     = new Network();
            $controller  = new Controller($backend, $network);
            $controller->blended_session_setter();
            $backend     = $this->getBackend($slug);
            $controller  = new Controller($backend, $network);
            if (empty($body['replaceList'])) {
                return true;
            }            
            $packageJptf = $controller->mergePackage($accountSlug, $slug, $body['replaceList']);            
        }
        catch (\Exception $e) {
            throw $e;
        }
        
        return $packageJptf;
    }

    /**
     * Get detail view of canonical version of a package.
     *
     * @param  string  $accountSlug is account slug.
     * @param  string  $slug        is package name.
     * @param  string  $sessionKey  is host session key.
     * @param  boolean $verbose     parameter to provide detail view of hyperlink objects.
     * @return array detail view of canonical version of a package.
     */    
    function get_canonicalDetails($accountSlug, $slug, $sessionKey, $verbose)
    {
        try {
            $backend     = $this->getBackend($slug);
            $network     = new Network();
            $controller  = new Controller($backend, $network);
            $controller->blended_session_setter();
            $backend     = $this->getBackend($slug);
            $controller  = new Controller($backend, $network);
            $packageJptf = $controller->canonicalDetails($accountSlug, $slug, $verbose);            
        }
        catch (\Exception $e) {
            throw $e;
        }
        
        return $packageJptf;
    }
    
    /**
     * Compare package.
     *
     * @param  string $accountSlug1 is the path to destination package.
     * @param  string $slug1        is package name.
     * @param  string $accountSlug2 is the path to source package.
     * @param  string $slug2        is package name.
     * @param  string $sessionKey   is host session key.
     * @return array package diifferences.
     */
    function get_compare_draft($accountSlug1, $slug1, $accountSlug2, $slug2, $sessionKey)
    {
        try {
            $backend     = $this->getBackend($slug);
            $network     = new Network();
            $controller  = new Controller($backend, $network);
            $controller->blended_session_setter();
            $backend     = $this->getBackend($slug1);
            $controller  = new Controller($backend, $network);
            $comparison  = $controller->compare_draft($accountSlug1, $slug1, $accountSlug2, $slug2);
        }
        catch (\Exception $e) {
            throw $e;
        }
        
        return $comparison;
    }
    
    /**
     * Compare canonical package version.
     *
     * @param  string $accountSlug1 is the path to destination package.
     * @param  string $slug1        is package name.
     * @param  string $accountSlug2 is the path to source package.
     * @param  string $slug2        is package name.
     * @param  string $sessionKey   is host session key.
     * @return array package diifferences.
     */
    function get_compare_canonical($accountSlug1, $slug1, $accountSlug2, $slug2, $sessionKey)
    {
        try {
            $backend     = $this->getBackend($slug);
            $network     = new Network();
            $controller  = new Controller($backend, $network);
            $controller->blended_session_setter();
            $backend     = $this->getBackend($slug1);
            $controller  = new Controller($backend, $network);
            $comparison  = $controller->compare_canonical($accountSlug1, $slug1, $accountSlug2, $slug2);
        }
        catch (\Exception $e) {
            throw $e;
        }
        
        return $comparison;
    }
    
    /**
     * Compare version package.
     *
     * @param  string $accountSlug1 is the path to destination package.
     * @param  string $slug1        is package name.
     * @param  string $accountSlug2 is the path to source package.
     * @param  string $slug2        is package name.
     * @param  string $label        is the package version.
     * @param  string $sessionKey   is host session key.
     * @return array package diifferences.
     */
    function get_compare_version($accountSlug1, $slug1, $accountSlug2, $slug2, $label, $sessionKey)
    {
        try {           
            $backend     = $this->getBackend($slug1);
            $network     = new Network();
            $controller  = new Controller($backend, $network);
            $controller->blended_session_setter();
            $backend     = $this->getBackend($slug1);
            $controller  = new Controller($backend, $network);
            $comparison  = $controller->compare_version($accountSlug1, $slug1, $accountSlug2, $slug2, $label);
        }
        catch (\Exception $e) {
            throw $e;
        }
        
        return $comparison;
    }
    
    /**
     * Retrieve file data in comparison of packages.
     *
     * @param  string $accountSlug1 is the path to destination package.
     * @param  string $slug1        is package name.
     * @param  string $accountSlug2 is the path to source package.
     * @param  string $slug2        is package name.
     * @param  string $filePath     is the file location.
     * @param  string $label        is the package version.
     * @param  string $sessionKey   is host session key.
     * @return array package diifferences.
     */
    function get_compare_labeledversion_file($accountSlug1, $slug1, $accountSlug2, $slug2, $filePath, $label, $sessionKey)
    {
        try {
            $backend     = $this->getBackend($slug);
            $network     = new Network();
            $controller  = new Controller($backend, $network);
            $controller->blended_session_setter();
            $backend     = $this->getBackend($slug1);
            $controller  = new Controller($backend, $network);
            $comparison  = $controller->compare_labeledversion_file($accountSlug1, $slug1, $accountSlug2, $slug2, $label, $filePath);
        }
        catch (\Exception $e) {
            throw $e;
        }
        
        return $comparison;
    }
    
    /**
     * Get package JPTF from filesystem or Hub.
     *
     * @param  string $accountSlug is account slug.
     * @param  string $slug        is package name.
     * @param  string $sessionKey  is host session key.
     * @return array package jptf.
     */
    function get_localDownload($accountSlug, $slug, $asHash, $sessionKey)
    {
        try {
            $backend     = $this->getBackend($slug);
            $network     = new Network();
            $controller  = new Controller($backend, $network);
            $sessionKey  = $controller->blended_session_setter();
            $backend     = $this->getBackend($slug);
            $controller  = new Controller($backend, $network);            
            $packageJptf = $controller->localDownload($accountSlug, $slug, $asHash, $sessionKey);
        }
        catch (\Exception $e) {
            throw $e;
        }
        
        return $packageJptf;
    }
    
    /**
     * Create empty package.
     *
     * @param  string $accountSlug is account slug.
     * @param  string $slug        is package name.
     * @param  array  $body        is request body.
     * @param  string $sessionKey  is host session key.
     * @return array package jptf.
     */
    function post_localCreate($accountSlug, $slug, $body, $sessionKey)
    {
        try {            
            $backend     = $this->getBackend($slug);
            $network     = new Network();
            $controller  = new Controller($backend, $network);
            $sessionKey = $controller->blended_session_setter();
            $backend     = $this->getBackend($slug);
            $controller  = new Controller($backend, $network);
            if (!isset($body['user'])) {
                $user = null;
            }
            if (!isset($body['version'])) {
                $version = "draft";
            }
            $packageJptf = $controller->localCreate($accountSlug, $slug, $version, $user, $sessionKey);
        }
        catch (\Exception $e) {
            throw $e;
        }
        
        return $packageJptf;
    }
    
    /**
     * Create a snapshot of a package.
     *
     * @param string $accountSlug is account slug.
     * @param string $slug        is package name.
     * @param array  $body        is request body.
     * @param string $sessionKey  is host session key.
     */
    function post_versionCreateSnapshot($accountSlug, $slug, $body, $sessionKey)
    {
        try {
            $backend     = $this->getBackend();
            $network        = new Network();
            $controller     = new Controller($backend, $network);
            $controller->blended_session_setter();
            $backend     = $this->getBackend($slug);
            $controller     = new Controller($backend, $network);
            $packageDetails = $controller->versionCreateSnapshot($accountSlug, $slug, $body);
        }
        catch (\Exception $e) {
            throw $e;
        }
        
        return $packageDetails;
    }
    
    /**
     * Get version data of a package in details.
     *
     * @param  string $accountSlug is account slug.
     * @param  string $slug        is package name.
     * @param  string $label       is the package version.
     * @param  string $sessionKey  is host session key.
     * @return array package details.
     */
    function get_getVersionDetails($accountSlug, $slug, $label, $sessionKey)
    {
        try {
            $backend     = $this->getBackend();
            $network        = new Network();
            $controller     = new Controller($backend, $network);
            $controller->blended_session_setter();
            $backend     = $this->getBackend($slug);
            $controller     = new Controller($backend, $network);
            $packageDetails = $controller->getVersionDetails($accountSlug, $slug, $label);
        }
        catch (\Exception $e) {
            throw $e;
        }
        
        return $packageDetails;
    }
    
    /**
     * Get version documents of a package.
     *
     * @param  string $accountSlug is account slug.
     * @param  string $slug        is package name.
     * @param  string $label       is the package version.
     * @param  string $sessionKey  is host session key.
     * @return array package jptf.
     */
    function get_versionDownload($accountSlug, $slug, $label, $asHash, $sessionKey)
    {
        try {          
            $backend     = $this->getBackend($slug);
            $network     = new Network();
            $controller  = new Controller($backend, $network);
            $controller->blended_session_setter();
            $backend     = $this->getBackend($slug);
            $controller  = new Controller($backend, $network);
            $packageJptf = $controller->versionDownload($accountSlug, $slug, $label, $asHash);
        }
        catch (\Exception $e) {
            throw $e;
        }
        
        return $packageJptf;
    }
    
    /**
     * Get version documents of a package in details.
     *
     * @param  string $accountSlug is account slug.
     * @param  string $slug        is package name.
     * @param  string $label       is the package version.
     * @param  string $filePath    is the file location.
     * @param  string $sessionKey  is host session key.
     * @return array package document jptf.
     */    
    function get_versionDownloadFile($accountSlug, $slug, $label, $filePath, $sessionKey)
    {
        try {
            $backend     = $this->getBackend($slug);
            $network     = new Network();
            $controller  = new Controller($backend, $network);
            $controller->blended_session_setter();
            $backend     = $this->getBackend($slug);
            $controller  = new Controller($backend, $network);
            $packageJptf = $controller->versionDownloadFile($accountSlug, $slug, $label, $filePath);
        }
        catch (\Exception $e) {
            throw $e;
        }
        
        return $packageJptf;
    }
    
    /**
     * Get file jptf package object.
     *
     * @param  string $accountSlug is account slug.
     * @param  string $slug        is package name.
     * @param  string $fileName    is the file location.
     * @param  string $sessionKey  is host session key.
     * @return array package document jptf.
     */
    function get_localDownloadFile($accountSlug, $slug, $fileName, $sessionKey)
    {
        try {
            $backend     = $this->getBackend($slug);
            $network     = new Network();
            $controller  = new Controller($backend, $network);
            $controller->blended_session_setter();
            $backend     = $this->getBackend($slug);
            $controller  = new Controller($backend, $network);
            $packageJptf = $controller->localDownloadFile($accountSlug, $slug, $fileName);
        }
        catch (\Exception $e) {
            throw $e;
        }
        
        return $packageJptf;
    }
    
    /**
     * Get package details either from Hub or local filesystem.
     *
     * @param  string $accountSlug is account slug.
     * @param  string $slug        is package name.
     * @param  string $sessionKey  is host session key.
     * @return array package detail.
     */
    function get_localPackageGet($accountSlug, $slug, $sessionKey)
    {
        try {
            $backend        = $this->getBackend();
            $network        = new Network();
            $controller     = new Controller($backend, $network);
            $sessionKey     = $controller->blended_session_setter();
            $backend        = $this->getBackend($slug);
            $controller     = new Controller($backend, $network);
            $packageDetails = $controller->localPackageGet($accountSlug, $slug);
        }
        catch (\Exception $e) {
            throw $e;
        }
        
        return $packageDetails;
    }
    
    /**
     * Delete package from local filesystem.
     *
     * @param string $accountSlug is account slug.
     * @param string $slug        is package name.
     * @param string $sessionKey  is host session key.
     */
    function delete_localDelete($accountSlug, $slug, $sessionKey)
    {
        try {
            $backend        = $this->getBackend();
            $network        = new Network();
            $controller     = new Controller($backend, $network);
            $controller->blended_session_setter();
            $backend        = $this->getBackend($slug);
            $controller     = new Controller($backend, $network);
            $controller->localDelete($accountSlug, $slug);
        }
        catch (\Exception $e) {
            throw $e;
        }
    }
    
    /**
     * Clone package on local file system.
     *
     * @param  string $accountSlug1 is the path to destination package.
     * @param  string $slug1        is package name.
     * @param  string $accountSlug2 is the path to source package.
     * @param  string $slug2        is package name.
     * @param  string $sessionKey   is host session key.
     * @return array package jptf.
     */
    function post_localClone($accountSlug1, $slug1, $accountSlug2, $slug2, $body, $sessionKey)
    {
        try {
            $backend     = $this->getBackend($slug1);
            $network     = new Network();
            $controller  = new Controller($backend, $network);
            $sessionKey = $controller->blended_session_setter();
            $status = $controller->localClone($accountSlug1, $slug1, $accountSlug2, $slug2, $body);
        }
        catch (\Exception $e) {
            throw $e;
        }
        
        return $status;
    }
    
    /**
     * Get file jptf package representation.
     *
     * @param  string $accountSlug is account slug.
     * @param  string $slug        is package name.
     * @param  string $filePath    is the file location.
     * @param  string $sessionKey  is host session key.
     * @return array package document jptf.
     */
    function get_getFile($accountSlug, $slug, $filePath, $sessionKey)
    {
        try {
            $backend     = $this->getBackend($slug);
            $network     = new Network();
            $controller  = new Controller($backend, $network);
            $controller->blended_session_setter();
            $backend     = $this->getBackend($slug);
            $controller  = new Controller($backend, $network);
            $packageJptf = $controller->getFile($filePath, $accountSlug, $slug);
        }
        catch (\Exception $e) {
            throw $e;
        }
        
        return $packageJptf;
    }
    
    /**
     * Get package object from Hub and save it to locally by calling controller get_package method.
     *
     * @param  string  $accountSlug is account slug.
     * @param  string  $slug        is package name.
     * @param  boolean $force       is required to pull package forcefylly.
     * @param  string  $asHash      is to get hash only package.
     * @param  string  $label       is the package version.
     * @param  string  $sessionKey  is host session key.
     * @return array package jptf.
     */
    function get_pull_package($accountSlug, $slug, $force, $asHash, $label, $sessionKey)
    {
        try {
            $backend      = $this->getBackend();
            $network      = new Network();
            $controller   = new Controller($backend, $network);
            $controller->blended_session_setter();
            $backend      = $this->getBackend($slug);
            $controller   = new Controller($backend, $network);            
            if (isset($label)) {
                $packageSlug = $accountSlug . '/' . $slug;
                $packageJptf = $controller->install_package($packageSlug, $label, true, true);
            } else if ($asHash) {
                $packageJptf = $controller->package_asHash($accountSlug, $slug, null, true);
            } else {
                if ($force == "force") {
                    $force = true;
                }
                $packageJptf = $controller->pull_package($slug, null, true, $force);
            }
        }
        catch (\Exception $e) {
            throw $e;
        }
        
        return $packageJptf;
    }

    /**
     * Get package hashes
     * @param string $accountSlug
     * @param string $slug
     * @param string $sessionKey
     * @return array
     * @throws \Exception
     */
    function get_getPackageHashes($accountSlug, $slug, $sessionKey) {
        try {            
            $network      = new Network();
            $backend      = $this->getBackend($slug);
            $controller   = new Controller($backend, $network); 
            $controller->blended_session_setter($sessionKey);
            $packageHash = $controller->getPackageHashes($accountSlug, $slug);
            return $packageHash;
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Push package clone to Hub.
     *
     * @param  string $accountSlug is account slug.
     * @param  string $slug        is package name.
     * @param  string $sessionKey  is host session key.
     * @return array package properties.
     */    
    function post_createHubPackage($accountSlug, $slug, $sessionKey)
    {
        try {
            $backend     = $this->getBackend($slug);
            $network     = new Network();
            $controller  = new Controller($backend, $network);
            $session     = $controller->blended_session_setter();
            $backend     = $this->getBackend($slug);
            $controller  = new Controller($backend, $network);
            $properties = $controller->createHubPackage($accountSlug, $slug, $session);
        }
        catch (\Exception $e) {
            throw $e;
        }
        
        return $properties;
    }
    
    /**
     * Push package object to Hub.
     *
     * @param  string $accountSlug is account slug.
     * @param  string $slug        is package name.
     * @param  array  $body        is request body.
     * @param  string $sessionKey  is host session key.
     * @return array as file differences conflict else partial jptf representation.
     */
    function put_push_package($accountSlug, $slug, $force, $pushToHub, $body, $sessionKey)
    {
        $backend     = $this->getBackend($slug);
        $network     = new Network();
        $controller  = new Controller($backend, $network);
        $controller->blended_session_setter();
        $backend     = $this->getBackend($slug);
        $controller  = new Controller($backend, $network);            
        if (isset($pushToHub) && $pushToHub) {
            $package = $controller->push_package($accountSlug, $slug, $force, $body['packageHash'], $body['replaceFromLocalList']);
            $jptf    = $package['jptf'];
            $package = $package['package'];
            $controller->save_packageHash($accountSlug . '/' . $slug, $package['package_hash']);                
            if (isset($package['tokens']) && !empty($package['tokens'])) {
                $result = $controller->uploadMedia($accountSlug, $slug, $package['tokens']);
                if (isset($result['data'])) {
                    $error = json_encode($result);
                    throw new \Exception($error);
                }
                $controller->save_packageHash($accountSlug . '/' . $slug, $result['packageHash']['package_hash']);
                return $package;
            } else {
                return $package;
            }
        } else {
            $package = $controller->push_package($accountSlug, $slug, $force);
        }
        
        return $package;
    }
    
    /**
     * Push package object to Hub.
     *
     * @param  string $accountSlug is account slug.
     * @param  string $slug        is package name.
     * @param  string $sessionKey  is host session key.
     * @param  object $error       is error object.
     * @return array as file differences conflict else partial jptf representation.
     */
    function put_push_package_helper($accountSlug, $slug, $sessionKey, $error) 
    {
        try {
            $errorString   = $error;
            $error         = json_decode($error, true);            
            if (isset($error['message']) && trim($error['message']) == "You cannot currently update this package in the Hub because another user is uploading files to the same package.  Please wait a few moments and try again.") {                 
                 throw new \Exception($errorString);
            } elseif ( isset($error['status_code']) && $error['status_code'] == 5111) {
                throw new \Exception($errorString);
            } elseif (isset($error['data']) && is_array($error['data'])) {
                 throw new \Exception($errorString);
            } elseif (isset($error['message']) && $error['message'] != "Draft has been changed. Please pull it first") {
                $network    = new Network();
                $backend    = $this->getBackend();
                $controller = new Controller($backend, $network);
                $controller->blended_session_setter();
                $backend    = $this->getBackend($slug);
                $controller = new Controller($backend, $network);            
                $package    = $controller->createatHub($accountSlug, $slug);
                if (isset($package['package_hash']) && !empty($package['package_hash'])) {
                    $controller->save_packageHash($accountSlug . '/' . $slug, $package['package_hash']);
                }
                if (isset($package['tokens']) && !empty($package['tokens'])) {
                    $result = $controller->uploadMedia($accountSlug, $slug, $package['tokens']);
                    if (isset($result['data'])) {
                        $error = json_encode($result);
                        throw new \Exception($error);
                    }
                }
                return $package;
            } else {
                $error['changemessage'] = true;
                $errorString = json_encode($error);
                throw new \Exception($errorString);
            }
        }
        catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Get package object from Hub and save it to locally by calling controller get_package method.
     *
     * @param  string $slug       is package name.
     * @param  string $sessionKey is host session key.
     * @return package jptf.
     */
    function put_pull_package($slug, $sessionKey)
    {
        try {
            $backend     = $this->getBackend($slug);
            $network     = new Network();
            $controller  = new Controller($backend, $network);
            $controller->blended_session_setter();
            $backend     = $this->getBackend($slug);
            $controller  = new Controller($backend, $network);
            $package     = $controller->pull_package($slug, null, true, true);
        }
        catch (\Exception $e) {
            throw $e;
        }
        
        return $package;
    }
    
    /**
     * Get package dependencies.
     *
     * @param  string  $slug       is package name.
     * @param  boolean $verbose    parameter to provide detail view of hyperlink objects.
     * @param  string  $sessionKey is host session key.
     * @return array package dependencies.
     */    
    function get_getDependencies($accountSlug, $slug, $verbose, $sessionKey)
    {
        try {
            $backend             = $this->getBackend();
            $network             = new Network();
            $controller          = new Controller($backend, $network);
            $controller->blended_session_setter();
            $packageDependencies = $controller->getDependencies($accountSlug, $slug, $verbose);
        }
        catch (\Exception $e) {
            throw $e;
        }
        
        return $packageDependencies;
    }
    
    /**
     * Access draft documents and directory.
     *
     * @param  string $accountSlug is account slug.
     * @param  string $slug        is package name.
     * @param  string $filePath    is the file location.
     * @param  string $sessionKey  is host session key.
     * @return array package detail.
     */
    function get_getDetailDocument($accountSlug, $slug, $filePath, $sessionKey)
    {
        try {
            $backend       = $this->getBackend();
            $network       = new Network();
            $controller    = new Controller($backend, $network);
            $controller->blended_session_setter();
            $packageDetail = $network->getDetailDocument($accountSlug, $slug, $filePath);
        }
        catch (\Exception $e) {
            throw $e;
        }
        
        return $packageDetail;
    }
    
    /**
     * Get package draft.
     *
     * @param  string $accountSlug is account slug.
     * @param  string $slug        is package name.
     * @param  string $sessionKey  is host session key.
     * @return array package jptf.
     */
    function get_getDraft($accountSlug, $slug, $sessionKey)
    {
        try {
            $backend     = $this->getBackend($slug);
            $network     = new Network();
            $controller  = new Controller($backend, $network);
            $controller->blended_session_setter();
            $packageJptf = $network->getDraft($accountSlug, $slug);
        }
        catch (\Exception $e) {
            throw $e;
        }
        
        return $packageJptf;
    }
    
    /**
     * Get package draft details.
     *
     * @param  string $accountSlug is account slug.
     * @param  string $slug        is package name.
     * @param  string $sessionKey  is host session key.
     * @return array package detail.
     */    
    function get_getPackageDetails($accountSlug, $slug, $sessionKey)
    {
        try {
            $backend      = $this->getBackend();
            $network      = new Network();
            $controller   = new Controller($backend, $network);
            $sessionKey = $controller->blended_session_setter();
            if ($sessionKey == "offline") {
                $packageDetail = $controller->get_Offline_Package_Details($slug, $accountSlug, $sessionKey);
            } else {
                try {
                    $packageDetail = $controller->getPackageDetails($accountSlug, $slug, true, $sessionKey);
                }
                catch (\Exception $e) {
                    $packageDetail = $controller->get_Offline_Package_Details($slug, $accountSlug, $sessionKey);
                }
            }
        }
        catch (\Exception $e) {
            throw $e;
        }
        
        return $packageDetail;
    }
    
    /**
     * Update package version as canonical version.
     *
     * @param string $accountSlug is account slug.
     * @param string $slug        is package name.
     * @param array  $body        is request body.
     * @param string $sessionKey  is host session key.
     */    
    function put_canonicalSet($accountSlug, $slug, $body, $sessionKey)
    {
        try {
            $backend     = $this->getBackend($slug);
            $network     = new Network();
            $controller  = new Controller($backend, $network);
            $controller->blended_session_setter();
            $controller->canonicalSet($accountSlug, $slug, $body);
        }
        catch (\Exception $e) {
            throw $e;
        }
    }
    
    /**
     * Update user draft package.
     *
     * @param  string $accountSlug is account slug.
     * @param  string $slug        is package name.
     * @param  array  $body        is request body.
     * @param  string $sessionKey  is host session key.
     * @return array package jptf.
     */    
    function put_updateDraft($accountSlug, $slug, $body, $sessionKey)
    {
        try {
            $backend      = $this->getBackend();
            $network      = new Network();
            $controller   = new Controller($backend, $network);
            $controller->blended_session_setter();
            $packageJptf  = $network->updateDraft($accountSlug, $slug, $body);
        }
        catch (\Exception $e) {
            throw $e;
        }
        
        return $packageJptf;
    }
    
    /**
     * Update Package.
     *
     * @param  string $accountSlug is account slug.
     * @param  string $slug        is package name.
     * @param  array  $body        is request body.
     * @param  string $sessionKey  is host session key.
     * @return array package jptf.
     */    
    function put_updatePackage($accountSlug, $slug, $body, $sessionKey)
    {
        try {
            $backend     = $this->getBackend($slug);
            $network     = new Network();
            $controller  = new Controller($backend, $network);
            $controller->blended_session_setter();
            $controller->updatePackage($accountSlug, $slug, $body);
        }
        catch (\Exception $e) {
            throw $e;
        }
        
        return $packageJptf;
    }
    
    /**
     * Detail View of user account.
     *
     * @param  string  $slug       is package name.
     * @param  boolean $verbose    parameter to provide detail view of hyperlink objects.
     * @param  string  $sessionKey is host session key.
     * @return array account details.
     */    
    function get_getAccountDetails($slug, $verbose, $sessionKey)
    {
        try {
            $backend     = $this->getBackend($slug);
            $network     = new Network();
            $controller  = new Controller($backend, $network);
            $controller->blended_session_setter();
            $accountDetails = $controller->getAccountDetails($slug, $verbose);
        }
        catch (\Exception $e) {
            throw $e;
        }
    
        return $accountDetails;
    }

    /**
     * Resend the inivite of shared package.
     *
     * @param  string $accountSlug is account slug.
     * @param  string $slug is package name.
     * @param  array  $body is request body.
     * @param  string $sessionKey is host session key.
     * @return array package details.
     */    
    function post_resendInvite($accountSlug, $slug, $body, $sessionKey)
    {
        try {
            $backend     = $this->getBackend($slug);
            $network     = new Network();
            $controller  = new Controller($backend, $network);
            $controller->blended_session_setter();
            $accountDetails = $controller->resendInvite($accountSlug, $slug, $body);
        }
        catch (\Exception $e) {
            throw $e;
        }
    
        return $accountDetails;
    }

    /**
     * Get list of user's account pending users.
     *
     * @param  string  $slug       is package name.
     * @param  boolean $verbose    parameter to provide detail view of hyperlink objects.
     * @param  string  $sessionKey is host session key.
     * @return array pending user.
     */    
    function get_getPendingUsers($slug, $verbose, $sessionKey)
    {
        try {
            $backend     = $this->getBackend($slug);
            $network     = new Network();
            $controller  = new Controller($backend, $network);
            $controller->blended_session_setter();
            $pendingUser = $controller->getPendingUsers($slug, $verbose);
        }
        catch (\Exception $e) {
            throw $e;
        }
        
        return $pendingUser;
    }
    
    /**
     * Invite user to join account.
     *
     * @param string $slug       is package name.
     * @param array  $body       is request body.
     * @param string $sessionKey is host session key.
     */    
    function post_inviteUser($slug, $body, $sessionKey)
    {
        try {
            $backend     = $this->getBackend($slug);
            $network     = new Network();
            $controller  = new Controller($backend, $network);
            $controller->blended_session_setter();
            $controller->inviteUser($slug, $body);
        }
        catch (\Exception $e) {
            throw $e;
        }
    }
    
    /**
     * Delete a pending user from user account.
     *
     * @param string $slug       is package name.
     * @param string $email      is user email address.
     * @param string $sessionKey is host session key.
     */
    function delete_removePendingUser($slug, $email, $sessionKey)
    {
        try {
            $backend     = $this->getBackend($slug);
            $network     = new Network();
            $controller  = new Controller($backend, $network);
            $controller->blended_session_setter();
            $controller->removePendingUser($email, $slug);
        }
        catch (\Exception $e) {
            throw $e;
        }
    }
    
    /**
     * Create a new user account.
     *
     * @param array   $body    is request body.
     * @param boolean $verbose parameter to provide detail view of hyperlink objects.
     */    
    function post_createAccount($body, $verbose)
    {
        try {
            $backend    = $this->getBackend();
            $network    = new Network();
            $controller = new Controller($backend, $network);
            $account = $controller->createAccount($body, $verbose);
        }
        catch (\Exception $e) {
            throw $e;
        }

        return $account;
    }    
    
    /**
     * Set Hub current account from Hub settings on to current stored configuration.
     *
     * @param string $accountSlug is account slug.
     * @param string $slug        is package name.
     * @param array  $body        is request body.
     * @param string $sessionKey  is host session key.
     */
    function put_setCurrentAccount($slug, $body, $sessionKey)
    {
        try {
            $backend      = $this->getBackend();
            $network      = new Network($sessionKey);
            $controller   = new Controller($backend, $network);
            $controller->blended_session_setter();
            $controller->setCurrentAccount($slug, $body);
        }
        catch (\Exception $e) {
            throw $e;
        }
    }
    
    /**
     * Get current logged-in account settings on to current stored configuration.
     *
     * @param  string  $slug       is package name.
     * @param  boolean $verbose    parameter to provide detail view of hyperlink objects.
     * @param  string  $sessionKey is host session key.
     * @return string user name.
     */
    function get_getCurrentAccount($slug, $verbose, $sessionKey)
    {
        try {
            $backend     = $this->getBackend($slug);
            $network     = new Network();
            $controller  = new Controller($backend, $network);
            $controller->blended_session_setter();
            $account     = $controller->getCurrentAccount($slug, $verbose);
        }
        catch (\Exception $e) {
            throw $e;
        }
        
        return $account;
    }

    /**
     * push image to hub.
     *
     * @param  string $slug       is package name.
     * @param  string $hashValue  is file hash.
     * @param  string $name       is image name.
     * @param  string $token      to upload image to hub.
     * @param  object $image      is Image Splfile object.
     * @param  string $sessionKey is host session key.
     * @return array updated packageHash after each image upload.
     */    
    function post_uploadMedia($slug, $hashValue, $name, $token, $image, $sessionKey)
    {
        try {
            $backend     = $this->getBackend($slug);
            $network     = new Network();
            $controller  = new Controller($backend, $network);
            $controller->blended_session_setter();
            $packageHash = $network->uploadMedia($slug, $hashValue, $name, $token, $image);
        }
        catch (\Exception $e) {
            throw $e;
        }
        
        return $packageHash;
    }
    


    /**
     * Get user account.
     *
     * @return array user account details.
     */
    function get_localaccountDetails()
    {
        try {
            $network    = new Network();
            $backend    = $this->getBackend();
            $controller = new Controller($backend, $network);
            $controller->blended_session_setter();
            $currentUser     = $controller->getCurrentUser();
            $currentAccount  = $controller->getCurrentAccount();
            $lastActiveAccount  = $controller->getLastActiveAccount();
            $userDetails     = array('current_user' => $currentUser, 'current_account'=> $currentAccount, 'last_active_account' => $lastActiveAccount);
        }
        catch (\Exception $e) {
            throw $e;
        }
        
        return $userDetails;
    }

    
    /**
     * Reset password of user account.
     *
     * @param  array  $body       is request body.
     * @param  string $sessionKey is host session key.
     * @return array account details.
     */
    function post_resetPassword($body, $sessionKey)
    {
        try {
            $backend     = $this->getBackend($slug);
            $network     = new Network();
            $controller  = new Controller($backend, $network);
            $controller->blended_session_setter();
            $controller->resetPassword($body);
        }
        catch (\Exception $e) {
            throw $e;
        }
    }
    
    /**
     * Update User Account.
     *
     * @param string $slug       is package name.
     * @param array  $body       is request body.
     * @param string $sessionKey is host session key.
     */    
    function put_updateAccount($slug, $body, $sessionKey)
    {
        try {
            $backend    = $this->getBackend();
            $network    = new Network();
            $controller = new Controller($backend, $network);
            $controller->blended_session_setter();
            $controller->updateAccount($slug, $body);
        }
        catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Resend Email Verification link to verifiy account if already not verified.
     *
     * @param string $sessionKey is host session key.
     * @param array  $body       is request body.
     */    
    function put_resendAccountVerificationEmail($sessionKey, $body)
    {
        try {
            $backend    = $this->getBackend();
            $network    = new Network();
            $controller = new Controller($backend, $network);
            $controller->blended_session_setter();
            $controller->resendAccountVerificationEmail($body);
        }
        catch (\Exception $e) {
            throw $e;
        }
    }
    
    /**
     * API to acquire package between accounts.
     *
     * @param  string $slug       is package name.
     * @param  array  $body       is request body.
     * @param  string $sessionKey is host session key.
     * @return array package jptf.
     */
    function post_acquirePackage($accountSlug, $slug, $body, $sessionKey)
    {
        try {
            $backend       = $this->getBackend();
            $network       = new Network();
            $controller    = new Controller($backend, $network);
            $sessionKey    = $controller->blended_session_setter();
            $acquireDetail = $controller->acquirePackage($accountSlug, $slug, $body);
        }
        catch (\Exception $e) {
            throw $e;
        }

        return $acquireDetail;
    }


    /**
     * API to accept package.
     * To share,transfer, get, buy, extend and clone a package.
     * @param string $accountSlug
     * @param string $slug
     * @param array $body
     * @param string $sessionKey
     * @return type
     * @throws Exception
     */
    function put_acceptAcquisition($accountSlug, $slug, $body, $sessionKey) {
        try {
            $backend      = $this->getBackend();
            $network      = new Network();
            $controller   = new Controller($backend, $network);
            $sessionKey   = $controller->blended_session_setter();
            $package      = $controller->acceptAcquisition($accountSlug, $slug, $body);
        }
        catch (\Exception $e) {
            throw $e;
        }
        
        return $package;
        
    }


    /**
     * API to reject shared package.
     *
     * @param  string $slug       is package name.
     * @param  array  $body       is request body.
     * @param  string $sessionKey is host session key.
     */
    function put_rejectAcquisitionInvite($accountSlug, $slug, $body, $sessionKey)
    {
        try {
            $backend      = $this->getBackend();
            $network      = new Network();
            $controller   = new Controller($backend, $network);
            $sessionKey   = $controller->blended_session_setter();
            $controller->rejectAcquisitionInvite($accountSlug, $slug, $body);
        }
        catch (\Exception $e) {
            throw $e;
        }        
    }

    /**
     * Search accounts by starting keywords of account name .
     *
     * @param string $accountSlug is account slug.
     * @param  string $sessionKey is host session key.
     * @return array user object.
     */
    function get_searchAccounts($accountSlug, $sessionKey)
    {
        try {
            $backend      = $this->getBackend();
            $network      = new Network();
            $controller   = new Controller($backend, $network);
            $sessionKey   = $controller->blended_session_setter();
            $userDetails      = $controller->searchAccounts($accountSlug);
        }
        catch (\Exception $e) {
            throw $e;
        }
        
        return $userDetails;
    }

    /**
     * Detail list of those accounts the package is shared or invited for accept share package.
     *
     * @param  string $slug       is package name.
     * @param  array  $body       is request body.
     * @param  string $sessionKey is host session key.
     * @return array package jptf.
     */
    function get_getAcquiredPackageAccountList($accountSlug, $slug, $sessionKey, $share)
    {
        try {
            $backend      = $this->getBackend();
            $network      = new Network();
            $controller   = new Controller($backend, $network);
            $sessionKey   = $controller->blended_session_setter();
            $package      = $controller->getAcquiredPackageAccountList($accountSlug, $slug, $share);
        }
        catch (\Exception $e) {
            throw $e;
        }
        
        return $package;
    }

    /**
     * Revoke shared pakage from user.
     *
     * @param  string $slug       is package name.
     * @param  array  $body       is request body.
     * @param  string $sessionKey is host session key.
     * @return array package jptf.
     */
    function delete_revokeAcquiredPackage($accountSlug, $slug, $body, $sessionKey)
    {
        try {
            $backend      = $this->getBackend();
            $network      = new Network();
            $controller   = new Controller($backend, $network);
            $sessionKey   = $controller->blended_session_setter();
            $package      = $controller->revokeAcquiredPackage($accountSlug, $slug, $body);
        }
        catch (\Exception $e) {
            throw $e;
        }
        
        return $package;
    }
    
    /**
     * Merge Hub and local stored packages details of logged-in user account.
     *
     * @param  string $sessionKey is host session key.
     * @return array package list detail.
     */    
    function get_sync_client_account($sessionKey, $share = "", $transfer_by_me = "", $transfer_to_me = "", $purchased = "", $snapshot = "", $dependency = "", $published = "", $owned_by_account = "", $org_packages = "") {
        try {
            $backend = $this->getBackend();
            $network = new Network();
            $controller = new Controller($backend, $network);
            $sessionKey = $controller->blended_session_setter($sessionKey);
            if ($sessionKey == "offline" && (!empty($share) || !empty($transfer_by_me) || !empty($transfer_to_me) || !empty($transfer_to_me) || !empty($transfer_to_me) || !empty($transfer_to_me || !empty($published) || !empty($owned_by_account) || !empty($org_packages)))) {
                $packages = array(
                    'items' => array()
                );
                return $packages;
            }
            if ($sessionKey == "offline" && (!empty($share) || !empty($purchased) || !empty($snapshot) || !empty($dependency) || !empty($published))) {
                $packages = array(
                    'items' => array()
                );
                return $packages;
            }
            if ($sessionKey == "offline") {
                $packages = array(
                    'items' => array()
                );
                $packages = $controller->get_local_Packages($packages);
                return $packages;
            }
            $accountSlug = $controller->getCurrentAccount();
            $package = $network->getPackages($accountSlug, $share, $transfer_by_me, $transfer_to_me, $purchased, $snapshot, $dependency, $published, $owned_by_account, $org_packages);
            $transfered_packages = $network->getPackages($accountSlug, "", 1);
            if ($transfered_packages) {
                foreach ($transfered_packages['items'] as $value) {
                    try {
                        //Filterd out published packages
                        if(!empty($published)) {
                            foreach ($package['items'] as $key => $publishedPackage) {
                                if($publishedPackage['slug'] == $value['slug']) {
                                    unset($package['items'][$key]);
                                    $package['items'] = array_values($package['items']);
                                }
                            }
                        }
                        //Compare with active theme too
                        global $ACTIVE_THEME;
                        $packageSlug = $accountSlug . "/" . $value['slug'];
                        if ($ACTIVE_THEME != $packageSlug)
                            $controller->localDelete($accountSlug, $value['slug'], null, true);
                    } catch (\Exception $e) {
                        
                    }
                }
            }
            if (is_array($package) && !empty($package) && empty($share) && empty($transfer_to_me) && empty($transfer_by_me) && empty($purchased) && empty($snapshot) && empty($dependency) && empty($published) && empty($owned_by_account) && empty($org_packages)) {
                $package = $controller->sync_client_account($package);
            }
        } catch (\Exception $e) {
            throw $e;
        }

        return $package;
    }

    /**
     * Get theme setting required to render theme.
     *
     * @return theme settings form stored configuration.
     */
    function get_get_theme_settings()
    {
        try {
            $backend       = $this->getBackend();
            $network       = new Network();
            $controller    = new Controller($backend, $network);
            $settingsTheme = $controller->get_theme_settings();
        }
        catch (\Exception $e) {
            throw $e;
        }
        
        return $settingsTheme;
    }
    
    /**
     * Save theme settings onto current configuration.
     *
     * @param  array $newSettings are setting to be saved.
     * @return array saved theme settings.
     */
    function put_save_theme_settings($newSettings)
    {
        try {
            $backend       = $this->getBackend();
            $network       = new Network();
            $controller    = new Controller($backend, $network);
            $settingsTheme = $controller->save_theme_settings($newSettings);            
        }
        catch (\Exception $e) {
            throw $e;
        }

        return $settingsTheme;
    }
    
    /**
     * Get WordPress specific setting required to render theme.
     *
     * @return array WordPress settings.
     */
    function get_get_wordpress_settings()
    {
        try {
            $backend           = $this->getBackend();
            $network           = new Network();
            $controller        = new Controller($backend, $network);
            $settingsWordpress = $controller->get_wordpress_settings();            
        }
        catch (\Exception $e) {
            throw $e;
        }

        return $settingsWordpress;
    }

    /**
     * Save WordPress settings onto current configuration.
     *
     * @param  array $newSettings are setting to be saved.
     * @return array saved WordPress settings.
     */        
    function put_save_wordpress_settings($newSettings)
    {
        try {
            $backend           = $this->getBackend();
            $network           = new Network();
            $controller        = new Controller($backend, $network);
            $settingsWordpress = $controller->save_wordpress_settings($newSettings);            
        }
        catch (\Exception $e) {
            throw $e;
        }

        return $settingsWordpress;
    }


    
    /**
     * Get navigation specific setting required to render theme.
     *
     * @param  string $sessionKey is host session key.
     * @return array navigation settings.
     */
    function get_get_navigation_settings()
    {
        try {            
            $backend           = $this->getBackend();
            $network           = new Network();
            $controller        = new Controller($backend, $network);
            $controller->blended_session_setter();
            $navigationSetting = $controller->get_navigation_settings();            
        }
        catch (\Exception $e) {
            throw $e;
        }

        return $navigationSetting;
    }

    /**
     * Save navigation settings onto current configuration.
     *
     * @param  array  $body       is request body.
     * @param  string $sessionKey is host session key.
     * @return array saved navigation settings.
     */
    function put_get_navigation_settings($body)
    {
        try {
            $backend           = $this->getBackend();
            $network           = new Network();
            $controller        = new Controller($backend, $network);
            $controller->blended_session_setter();
            $navigationSetting = $controller->save_navigation_settings($body);            
        }
        catch (\Exception $e) {
            throw $e;
        }

        return $navigationSetting;
    }
    
    /**
     * Get Hub Specific setting for getting and putting updates to user.
     *
     * @return array hub settings.
     */
    function get_get_hub_settings()
    {
        try {
            $backend       = $this->getBackend();
            $network       = new Network();
            $controller    = new Controller($backend, $network);
            $updateSetting = $controller->get_hub_settings();            
        }
        catch (\Exception $e) {
            throw $e;
        }

        return $updateSetting;
    }
    
    /**
     * Save Hub settings onto current configuration to manage local packages.
     *
     * @param  array $body is request body.
     * @return array saved Hub settings.
     */
    function put_save_hub_settings($body)
    {
        try {
            $backend       = $this->getBackend();
            $network       = new Network();
            $controller    = new Controller($backend, $network);
            $updateSetting = $controller->save_hub_settings($body);
        }
        catch (\Exception $e) {
            throw $e;
        }

        return $updateSetting;
    }
    
    /**
     * Retrieve all accounts from the system that the user has access to.
     *
     * @param  string  $slug       is package name.
     * @param  boolean $verbose    parameter to provide detail view of hyperlink objects.
     * @param  string  $sessionKey is host session key.
     * @return array all member accounts list.
     */
    function get_getAccountList($slug, $verbose, $sessionKey)
    {
        try {
            $backend     = $this->getBackend($slug);
            $network     = new Network();
            $controller  = new Controller($backend, $network);
            $controller->blended_session_setter();
            $backend     = $this->getBackend($slug);
            $controller  = new Controller($backend, $network);
            $accountList = $controller->getAccountList($slug, $verbose);            
        }
        catch (\Exception $e) {
            throw $e;
        }

        return $accountList;
    }

    /**
     * Get Details of package rating submitted by any user.
     *
     * @param  string  $accountSlug is account slug.
     * @param  string  $slug        is package name.
     * @param  string  $ratingPk    is unique id
     * @param  boolean $verbose     parameter to provide detail view of hyperlink objects.
     * @param  string  $sessionKey  is host session key.
     * @return array rating details.
     */    
    function get_getRatingDetails($accountSlug, $slug, $ratingPk, $verbose, $sessionKey)
    {
        try {
            $backend     = $this->getBackend($slug);
            $network     = new Network();
            $controller  = new Controller($backend, $network);
            $controller->blended_session_setter();
            $backend     = $this->getBackend($slug);
            $controller  = new Controller($backend, $network);
            $getRating   = $controller->getRatingDetails($accountSlug, $slug, $ratingPk, $verbose);
            
            return $getRating;
        }
        catch (\Exception $e) {
            throw $e;
        }
    }
    
    /**
     * Get Details of package rating submitted by any user.
     *
     * @param  string $accountSlug is account slug.
     * @param  string $slug        is package name.
     * @param  string $ratingPk    is unique id
     * @param  array  $body        is request body.
     * @param  string $sessionKey  is host session key.
     * @return array rating details.
     */
    function put_updateRating($accountSlug, $slug, $ratingPk, $body, $sessionKey)
    {
        try {
            $backend     = $this->getBackend($slug);
            $network     = new Network();
            $controller  = new Controller($backend, $network);
            $controller->blended_session_setter();
            $backend     = $this->getBackend($slug);
            $controller  = new Controller($backend, $network);
            $controller->updateRating($accountSlug, $slug, $ratingPk, $body);
        }
        catch (\Exception $e) {
            throw $e;
        }
    }
    
    /**
     * Get list of licenses associated with package.
     *
     * @param  string $accountSlug is account slug.
     * @param  string $slug        is package name.
     * @param  string $label       is the package version.
     * @param  string $sessionKey  is host session key.
     * @return array license details.
     */    
    function get_getLicenses($accountSlug, $slug, $label, $sessionKey)
    {
        try {
            $backend     = $this->getBackend();
            $network        = new Network();
            $controller     = new Controller($backend, $network);
            $controller->blended_session_setter();
            $backend     = $this->getBackend($slug);
            $controller     = new Controller($backend, $network);
            $licenseDetails = $controller->getLicenses($accountSlug, $slug, $label);
            
            return $licenseDetails;
        }
        catch (\Exception $e) {
            throw $e;
        }
    }
    
    /**
     * Update price of package license.
     *
     * @param string $accountSlug is account slug.
     * @param string $slug        is package name.
     * @param string $label       is the package version.
     * @param string $name        is name of the license.
     * @param array  $body        is request body.
     * @param string $sessionKey  is host session key.
     */
    function put_updateLicense($accountSlug, $slug, $label, $name, $body, $sessionKey)
    {
        try {
            $backend     = $this->getBackend();
            $network        = new Network();
            $controller     = new Controller($backend, $network);
            $controller->blended_session_setter();
            $backend     = $this->getBackend($slug);
            $controller     = new Controller($backend, $network);
            $controller->updateLicense($accountSlug, $slug, $label, $name, $body);
        }
        catch (\Exception $e) {
            throw $e;
        }
    }
    
    /**
     * delete a package license.
     *
     * @param string $accountSlug is account slug.
     * @param string $slug        is package name.
     * @param string $label       is the package version.
     * @param string $name        is name of the license.
     * @param string $sessionKey  is host session key.
     */    
    function delete_removeLicense($accountSlug, $slug, $label, $name, $sessionKey)
    {
        try {
            $backend     = $this->getBackend($slug);
            $network     = new Network();
            $controller  = new Controller($backend, $network);
            $controller->blended_session_setter();
            $backend     = $this->getBackend($slug);
            $controller  = new Controller($backend, $network);
            $controller->removeLicense($accountSlug, $slug, $label, $name);
        }
        catch (\Exception $e) {
            throw $e;
        }
    }
    
    /**
     * Retrieve list of package version.
     *
     * @param  string $accountSlug is account slug.
     * @param  string $slug        is package name.
     * @param  string $sessionKey  is host session key.
     * @return array list of package version.
     */    
    function get_getVersions($accountSlug, $slug, $sessionKey)
    {
        try {
            $backend     = $this->getBackend($slug);
            $network     = new Network();
            $controller  = new Controller($backend, $network);
            $controller->blended_session_setter();
            $versionList = $controller->getVersions($accountSlug, $slug);
            
            return $versionList;
        }
        catch (\Exception $e) {
            throw $e;
        }
    }
    
    /**
     * Clone draft on local file system .
     *
     * @param  string $accountSlug1 is the path to destination package.
     * @param  string $slug1        is package name.
     * @param  string $accountSlug2 is the path to source package.
     * @param  string $slug2        is package name.
     * @param  string $sessionKey   is host session key.
     * @return array list of package version.
     */
    function post_draftClone($accountSlug1, $slug1, $accountSlug2, $slug2, $sessionKey)
    {
        try {
            $backend      = $this->getBackend();
            $network      = new Network();
            $controller   = new Controller($backend, $network);
            $controller->blended_session_setter();
            $backend      = $this->getBackend($slug1);
            $controller   = new Controller($backend, $network);
            $packageJptf  = $controller->draftClone($accountSlug1, $slug1, $accountSlug2, $slug2);
            
            return $packageJptf;
        }
        catch (\Exception $e) {
            throw $e;
        }
    }
    
    /**
     * Clone version of a package on local file system .
     *
     * @param  string $accountSlug1 is the path to destination package.
     * @param  string $slug1        is package name.
     * @param  string $accountSlug2 is the path to source package.
     * @param  string $slug2        is package name.
     * @param  string $label        is the package version.
     * @param  string $sessionKey   is host session key.
     * @return array package jptf representation.
     */
    function post_versionClone($accountSlug1, $slug1, $accountSlug2, $slug2, $label, $packageHash, $body, $sessionKey)
    {
        try {
            $backend      = $this->getBackend();
            $network      = new Network();
            $controller   = new Controller($backend, $network);
            $controller->blended_session_setter();
            $backend      = $this->getBackend($slug1);
            $controller   = new Controller($backend, $network);
            $packageJptf  = $controller->versionClone($accountSlug1, $slug1, $accountSlug2, $slug2, $label, $packageHash, $body);
            
            return $packageJptf;
        }
        catch (\Exception $e) {
            throw $e;
        }
    }
    
    /**
     * Compare draft document.
     *
     * @param  string $accountSlug1 is the path to destination package.
     * @param  string $slug1        is package name.
     * @param  string $accountSlug2 is the path to source package.
     * @param  string $slug2        is package name.
     * @param  string $filePath     is the file location.
     * @param  string $sessionKey   is host session key.
     * @return array both file differences.
     */
    function get_compareFileWithDraft($accountSlug1, $slug1, $accountSlug2, $slug2, $filePath, $sessionKey)
    {
        try {
            $backend     = $this->getBackend();
            $network        = new Network();
            $controller     = new Controller($backend, $network);
            $controller->blended_session_setter();
            $backend        = $this->getBackend($slug1);
            $controller     = new Controller($backend, $network);
            $fileComparison = $controller->compareFileWithDraft($filePath, $accountSlug1, $slug1, $accountSlug2, $slug2);
            
            return $fileComparison;
        }
        catch (\Exception $e) {
            throw $e;
        }
    }
    
    /**
     * Compare draft document.
     *
     * @param  string $accountSlug1 is the path to destination package.
     * @param  string $slug1        is package name.
     * @param  string $accountSlug2 is the path to source package.
     * @param  string $slug2        is package name.
     * @param  string $filePath     is the file location.
     * @param  string $sessionKey   is host session key.
     * @return array both file differences.
     */
    function get_compareFileWithLocal($accountSlug1, $slug1, $accountSlug2, $slug2, $filePath, $sessionKey)
    {
        try {
            $backend     = $this->getBackend();
            $network        = new Network();
            $controller     = new Controller($backend, $network);
            $controller->blended_session_setter();
            $backend        = $this->getBackend($slug1);
            $controller     = new Controller($backend, $network);
            $fileComparison = $controller->compareFileWithLocal($filePath, $accountSlug1, $slug1, $accountSlug2, $slug2);
            
            return $fileComparison;
        }
        catch (\Exception $e) {
            throw $e;
        }
    }
    
    /**
     * Compare version document.
     *
     * @param  string $accountSlug1 is the path to destination package.
     * @param  string $slug1        is package name.
     * @param  string $accountSlug2 is the path to source package.
     * @param  string $slug2        is package name.
     * @param  string $label        is the package version.
     * @param  string $filePath     is the file location.
     * @param  string $sessionKey   is host session key.
     * @return array both file differences.
     */
    function get_compareFileWithVersion($accountSlug1, $slug1, $accountSlug2, $slug2, $label, $filePath, $sessionKey)
    {
        try {
            $backend     = $this->getBackend();
            $network        = new Network();
            $controller     = new Controller($backend, $network);
            $controller->blended_session_setter();
            $backend        = $this->getBackend($slug1);
            $controller     = new Controller($backend, $network);
            $fileComparison = $controller->compareFileWithVersion($filePath, $accountSlug1, $slug1, $accountSlug2, $slug2, $label);
            
            return $fileComparison;
        }
        catch (\Exception $e) {
            throw $e;
        }
    }
    
    /**
     * Compare local document.
     *
     * @param  string $accountSlug1 is the path to destination package.
     * @param  string $slug1        is package name.
     * @param  string $accountSlug2 is the path to source package.
     * @param  string $slug2        is package name.
     * @param  string $sessionKey   is host session key.
     * @return array both file differences.
     */    
    function get_compareWithLocal($accountSlug1, $slug1, $accountSlug2, $slug2, $sessionKey)
    {
        try {
            $backend           = $this->getBackend();
            $network           = new Network();
            $controller        = new Controller($backend, $network);
            $sessionKey        = $controller->blended_session_setter();
            $backend           = $this->getBackend($slug1);
            $controller        = new Controller($backend, $network);
            $packageComparison = $controller->compareWithLocal($accountSlug1, $slug1, $accountSlug2, $slug2);
            
            return $packageComparison;
        }
        catch (\Exception $e) {
            throw $e;
        }
    }
    
    /**
     * Update draft.
     *
     * @param  string $accountSlug1 is the path to destination package.
     * @param  string $slug1        is package name.
     * @param  string $accountSlug2 is the path to source package.
     * @param  string $slug2        is package name.
     * @param  string $sessionKey   is host session key.
     * @return array draft jptf representation.
     */
    function put_updateFromDraft($accountSlug1, $slug1, $accountSlug2, $slug2, $sessionKey)
    {
        try {
            $backend      = $this->getBackend();
            $network      = new Network();
            $controller   = new Controller($backend, $network);
            $controller->blended_session_setter();
            $backend      = $this->getBackend($slug1);
            $controller   = new Controller($backend, $network);
            $packageJptf  = $controller->updateFromDraft($accountSlug1, $slug1, $accountSlug2, $slug2);
            
            return $packageJptf;
        }
        catch (\Exception $e) {
            throw $e;
        }
    }
    
    /**
     * Update local draft.
     *
     * @param  string $accountSlug1 is the path to destination package.
     * @param  string $slug1        is package name.
     * @param  string $accountSlug2 is the path to source package.
     * @param  string $slug2        is package name.
     * @param  string $sessionKey   is host session key.
     * @return boolean status.
     */    
    function put_updateFromLocal($accountSlug1, $slug1, $accountSlug2, $slug2, $files, $sessionKey)
    {
        try {
            $backend      = $this->getBackend();
            $network      = new Network();
            $controller   = new Controller($backend, $network);
            $controller->blended_session_setter();
            $backend      = $this->getBackend($slug1);
            $controller   = new Controller($backend, $network);
            $packageJptf  = $controller->updateFromLocal($accountSlug1, $slug1, $accountSlug2, $slug2, $files);
            
            return $packageJptf;
        }
        catch (\Exception $e) {
            throw $e;
        }
    }
    
    /**
     * Update draft from version document.
     *
     * @param  string $accountSlug1 is the path to destination package.
     * @param  string $slug1        is package name.
     * @param  string $accountSlug2 is the path to source package.
     * @param  string $slug2        is package name.
     * @param  string $label        is the package version.
     * @param  string $sessionKey   is host session key.
     * @return boolean status.
     */
    function put_updateFromVersion($accountSlug1, $slug1, $accountSlug2, $slug2, $label, $files, $sessionKey)
    {
        try {
            $backend      = $this->getBackend();
            $network      = new Network();
            $controller   = new Controller($backend, $network);
            $controller->blended_session_setter();
            $backend      = $this->getBackend($slug1);
            $controller   = new Controller($backend, $network);
            $packageJptf  = $controller->updateFromVersion($accountSlug1, $slug1, $accountSlug2, $slug2, $label, $files);
            
            return $packageJptf;
        }
        catch (\Exception $e) {
            throw $e;
        }
    }
    
    /**
     * Get canonical version of a package and store it in local filesystem.
     *
     * @param  string $accountSlug is account slug.
     * @param  string $slug        is package name.
     * @param  string $sessionKey  is host session key.
     * @return array package Jptf representation.
     */
    function get_canonicalDownload($accountSlug, $slug, $sessionKey)
    {
        try {
            $backend     = $this->getBackend($slug);
            $network     = new Network();
            $controller  = new Controller($backend, $network);
            $controller->blended_session_setter();
            $backend     = $this->getBackend($slug);
            $controller  = new Controller($backend, $network);
            $packageJptf = $controller->canonicalDownload($accountSlug, $slug);
            
            return $packageJptf;
        }
        catch (\Exception $e) {
            throw $e;
        }
    }
    
    /**
     * Update draft from canonical document.
     *
     * @param  string $accountSlug1 is the path to destination package.
     * @param  string $slug1        is package name.
     * @param  string $accountSlug2 is the path to source package.
     * @param  string $slug2        is package name.
     * @param  string $sessionKey   is host session key.
     * @return array package jptf representation.
     */
    function put_updateFromCanonical($accountSlug1, $slug1, $accountSlug2, $slug2, $sessionKey)
    {
        try {
            $backend      = $this->getBackend();
            $network      = new Network();
            $controller   = new Controller($backend, $network);
            $controller->blended_session_setter();
            $backend      = $this->getBackend($slug1);
            $controller   = new Controller($backend, $network);
            $packageJptf  = $controller->updateFromCanonical($accountSlug1, $slug1, $accountSlug2, $slug2);
            
            return $packageJptf;
        }
        catch (\Exception $e) {
            throw $e;
        }
    }
    
    /**
     * Clone version of a package on local file system .
     *
     * @param  string $accountSlug1 is the path to destination package.
     * @param  string $slug1        is package name.
     * @param  string $accountSlug2 is the path to source package.
     * @param  string $slug2        is package name.
     * @param  string $sessionKey   is host session key.
     * @return array package jptf representation.
     */    
    function post_canonicalClone($accountSlug1, $slug1, $accountSlug2, $slug2, $body, $sessionKey)
    {
        try {
            $backend      = $this->getBackend();
            $network      = new Network();
            $controller   = new Controller($backend, $network);
            $controller->blended_session_setter();
            $backend      = $this->getBackend($slug1);
            $controller   = new Controller($backend, $network);
            $packageJptf  = $controller->canonicalClone($accountSlug1, $slug1, $accountSlug2, $slug2, $body);
            
            return $packageJptf;
        }
        catch (\Exception $e) {
            throw $e;
        }
    }
    
    /**
     * Compare Canonical version of a document.
     *
     * @param  string $accountSlug1 is the path to destination package.
     * @param  string $slug1        is package name.
     * @param  string $accountSlug2 is the path to source package.
     * @param  string $slug2        is package name.
     * @param  string $filePath     is the file location.
     * @param  string $sessionKey   is host session key.
     * @return array both file differences.
     */
    function get_compareFileWithCanonical($accountSlug1, $slug1, $accountSlug2, $slug2, $filePath, $sessionKey)
    {
        try {
            $backend      = $this->getBackend();
            $network      = new Network();
            $controller   = new Controller($backend, $network);
            $controller->blended_session_setter();
            $backend      = $this->getBackend($accountSlug1 . '/' . $slug1);
            $controller   = new Controller($backend, $network);
            $packageJptf  = $controller->compareFileWithCanonical($filePath, $accountSlug1, $slug1, $accountSlug2, $slug2);
            
            return $packageJptf;
        }
        catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * API to acquire package between accounts.
     *
     * @param  string $accountSlug is the path to destination package.
     * @param  string $slug is package name.
     * @param  string $sessionKey   is host session key.
     * @return array acquire package details.
     */
    function get_sharedPackageListGet($accountSlug, $slug, $sessionKey)
    {
        try {
            $backend      = $this->getBackend();
            $network      = new Network();
            $controller   = new Controller($backend, $network);
            $controller->blended_session_setter();
            $backend      = $this->getBackend($accountSlug1 . '/' . $slug1);
            $controller   = new Controller($backend, $network);
            $packageJptf  = $controller->sharedPackageListGet($accountSlug, $slug);
            
            return $packageJptf;
        }
        catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * API to get package publications
     * 
     * @param string $accountSlug
     * @param string $slug
     * @param string $sessionKey
     * @return array package publications.
     * @throws \Exception
     */
    function get_publication($accountSlug, $slug, $sessionKey) {
        try {
            $backend      = $this->getBackend();
            $network      = new Network();
            $controller   = new Controller($backend, $network);
            $controller->blended_session_setter();
            $backend      = $this->getBackend($accountSlug1 . '/' . $slug1);
            $controller   = new Controller($backend, $network);
            $packageJptf  = $controller->getPublication($accountSlug, $slug);
            
            return $packageJptf;
        }
        catch (\Exception $e) {
            throw $e;
        }
        
    }
    
    /**
     * API to update package publications
     * 
     * @param string $accountSlug
     * @param string $slug
     * @param string $sessionKey
     * @param array $body
     * @return array package publications.
     * @throws \Exception
     */
    function put_publications($accountSlug, $slug, $commit, $body, $sessionKey) {
        try {
            $backend      = $this->getBackend();
            $network      = new Network();
            $controller   = new Controller($backend, $network);
            $controller->blended_session_setter();
            $backend      = $this->getBackend($accountSlug1 . '/' . $slug1);
            $controller   = new Controller($backend, $network);
            $packageJptf  = $controller->updatePublication($accountSlug, $slug, $commit, $body);
            
            return $packageJptf;
        }
        catch (\Exception $e) {
            throw $e;
        }
        
    }
    
    
    /**
     * API to validate package publications
     * 
     * @param string $accountSlug
     * @param string $slug
     * @param string $sessionKey
     * @param array $body
     * @return array package publications.
     * @throws \Exception
     * @todo check return type after React test
     */
    function post_validatePublication($accountSlug, $slug, $sessionKey, $body) {
        try {
            $backend      = $this->getBackend();
            $network      = new Network();
            $controller   = new Controller($backend, $network);
            $controller->blended_session_setter();
            $backend      = $this->getBackend($accountSlug1 . '/' . $slug1);
            $controller   = new Controller($backend, $network);
            $packageJptf  = $controller->validatePublication($accountSlug, $slug, $body);
            
            return $packageJptf;
        }
        catch (\Exception $e) {
            throw $e;
        }
    }
    
    /**
     * API to get package version bundles
     * 
     * @param string $accountSlug
     * @param string $slug
     * @param string $sessionKey
     * @return array package version bundles
     * @throws \Exception
     */
    function get_versionBundles($accountSlug, $slug, $sessionKey) {
        
        try {
            $backend      = $this->getBackend();
            $network      = new Network();
            $controller   = new Controller($backend, $network);
            $controller->blended_session_setter();
            $backend      = $this->getBackend($accountSlug1 . '/' . $slug1);
            $controller   = new Controller($backend, $network);
            $packageJptf  = $controller->getVersionBundles($accountSlug, $slug);
            
            return $packageJptf;
        }
        catch (\Exception $e) {
            throw $e;
        }
        
    }

    /**
     * API to get package version bundles
     * 
     * @param string $accountSlug
     * @param string $slug
     * @param string $sessionKey
     * @throws \Exception
     */
    function get_currenthash($accountSlug, $slug, $sessionKey) {
        
        try {
            $backend      = $this->getBackend();
            $network      = new Network();
            $controller   = new Controller($backend, $network);
            $controller->blended_session_setter();
            $backend      = $this->getBackend($accountSlug1 . '/' . $slug1);
            $controller   = new Controller($backend, $network);
            $packageHash  = $controller->currenthash($accountSlug, $slug);
            
            return $packageHash;
        }
        catch (\Exception $e) {
            throw $e;
        }
        
    }
    
    /**
     * API to get media files against hash
     * @param string $sessionKey
     * @param array $body
     * @return array
     * @throws \Exception
     */
    function post_getPackageMedia($sessionKey, $body) {
        try {
            $backend      = $this->getBackend();
            $network      = new Network();
            $controller   = new Controller($backend, $network);
            $controller->blended_session_setter();
            $media  = $controller->getPackageMedia($body);
            return $media;
        }
        catch (\Exception $e) {
            throw $e;
        }
    }
    
}
    
