<?php
/**
 * Connects Blended Hub for all package related calls.
 * Also keeps Hub session in its constructor variable.
 *
 * @blended
 */

namespace Blended\hostlib;

// Communicate to Hub via Swagger
require_once 'SwaggerClient-php/autoload.php';

use Swagger\Client\ApiException;

/**
 * Request Hub for retrieving and making changes to package.
 * All package and account Api's are defined within network.
 */
class Network
{
    /**
     * What the Hub SessionKey.
     *
     * @access public
     * @var    string
     */
    public $sessionKey;
    
    /**
     * PackagesApi Instance
     * @var object
     */
    protected $packageInstance;
    
    /**
     * AccountsApi Instance
     * @var object
     */
    protected $accountInstance;

    /**
     * Retrieve Intermediary representation of package directory.
     *
     * @param string $dir                is the subfolder of package directory.
     * @param object $package[optional].
     */
    function __construct($sessionKey = null)
    {
        $this->sessionKey = $sessionKey;
        $this->accountInstance = new \Swagger\Client\Api\AccountsApi();
        $this->packageInstance = new \Swagger\Client\Api\PackagesApi();
    }
    
    /**
     * Login to Hub.
     *
     * @param  string $username is the username.
     * @param  string $password is user password.
     * @return hub session key.
     */
    function login($username, $password, $active_account = "")
    {
        $apiInstance = $this->accountInstance;
        $body         = new \Swagger\Client\Model\Login(array("user_name" => $username,"password" => $password, "active_account" => $active_account));        
        try {
            $result = $apiInstance->login($body);
            $result = json_decode($result, true);
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }

        return $result;            
    }
    
    /**
     * Logout from Hub.
     */
    function logout()
    {
        $apiInstance = $this->accountInstance;
        $sessionKey  = $this->sessionKey;
        $body         = array('sessionkey' => $sessionKey);
        $body = new \Swagger\Client\Model\Logout($body); // \Swagger\Client\Model\Logout | The request data used to logout.        
        try {
            $apiInstance->logout($body);            
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }
    }
    
    /**
     * Request used to add admin user into account.
     *
     * @param string $slug is account unique indentifier.
     * @param array  $body is the content of the request used to add admin into account.
     */
    function addAccountAdmin($slug, $body)
    {
        $apiInstance = $this->accountInstance;
        $sessionKey  = $this->sessionKey;
        $body         = new \Swagger\Client\Model\AddAdmin($body);
        try {
            $apiInstance->addAccountAdmin($slug, $sessionKey, $body);
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }
    }

    /**
     * Add a regular user into account.
     *
     * @param string $slug is account unique indentifier.
     * @param array  $body is the content of the request used to add admin into account.
     */
    function addAccountUser($slug, $body)
    {
        $apiInstance = $this->accountInstance;
        $sessionKey  = $this->sessionKey;
        $body         = new \Swagger\Client\Model\AddAdmin($body);
        try {
            $apiInstance->addAccountUser($slug, $sessionKey, $body);
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }
    }
    
    /**
     * Get user packages from a Hub account.
     *
     * @param  string $accountSlug is user account name.
     * @return object package jptf.
     */
    function getPackages($accountSlug, $share = "", $transfer_by_me = "", $transfer_to_me = "", $purchased = "", $snapshot = "", $dependency = "", $published = "", $owned_by_account = "", $org_packages = "")
    {
        $apiInstance = $this->packageInstance;
        try {
            $sessionKey = $this->sessionKey;
            $verbose    = false;
            $result     = $apiInstance->getPackages($accountSlug, $sessionKey, $verbose, $share, $snapshot, $dependency, $purchased, $transfer_by_me, $transfer_to_me, $published, $owned_by_account, $org_packages);
            $result = $result->__toString();
            $result = json_decode($result, true);
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }

        return $result;
    }
    
    /**
     * Get user packages from a Hub account.
     *
     * @param  string $accountSlug is user account name.
     * @param  string $slug        is slug of package to return.
     * @param  string $verbose     parameter to provide detail view of hyperlink objects.
     * @return object blended package object.
     */
    function getPackageDetails($accountSlug, $slug, $verbose)
    {
        $apiInstance = $this->packageInstance;
        $sessionKey  = $this->sessionKey;  
        
        try {
            $result = $apiInstance->getPackageDetails($accountSlug, $slug, $sessionKey, $verbose);
            $result = $result->__toString();
            $result = json_decode($result, true);
           
        }
        catch (ApiException $e) {            
            throw new \Exception($e->getResponseObject());
        }

        return $result;
    }
    
    /**
     * Get user packages from a Hub account.
     *
     * @param  string $accountSlug is user account name.
     * @param  string $slug        is slug of package to return.
     * @param  string $asHashes    value used to return only hashes
     * @return object blended package object.
     */
    function downloadDraft($accountSlug, $slug, $asHashes = "0")
    {
        $apiInstance = $this->packageInstance;
        $sessionKey  = $this->sessionKey;
        
        try {
            $result = $apiInstance->downloadDraft($accountSlug, $slug, $sessionKey, $asHashes);
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }

        return $result;
    }
    
    /**
     * Get version documents
     *
     * @param  string $accountSlug is user account name.
     * @param  string $slug        is slug of package to return.
     * @param  string $label       value used to uniquely identify version.
     * @return object blended package object.
     */
    function download($accountSlug, $slug, $label, $asHashes = "0")
    {
        $apiInstance = $this->packageInstance;
        $sessionKey  = $this->sessionKey;
        try {
            $result = $apiInstance->download($accountSlug, $slug, $label, $sessionKey, $asHashes);
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }

        return $result;
    }
    
    /**
     * Add a rating to a package.
     *
     * @param string $slug is slug of package to return.
     * @param array  $body is the request data used to add package rating.     
     */
    function addRating($slug, $body)
    {
        $apiInstance = $this->packageInstance;
        $sessionKey  = $this->sessionKey;
        $body         = new \Swagger\Client\Model\AddRating($body); // \Swagger\Client\Model\AddRating | request data used to add package rating.        
        try {
            $result = $apiInstance->addRating($slug, $sessionKey, $body);
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }
    }

    /**
     * Returns list of ratings submitted by users for a package.
     *
     * @param string $slug    is slug of package to return.
     * @param string $verbose parameter to provide detail view of hyperlink objects.
     */    
    function getRatings($slug, $verbose)
    {
        $apiInstance = $this->packageInstance;
        $sessionKey  = $this->sessionKey;        
        try {
            $result = $apiInstance->getRatings($slug, $sessionKey, $verbose);
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }

        return $result;
    }
    
    /**
     * Get Details of package rating submitted by any user.
     *
     * @param  string $accountSlug is user account name.
     * @param  string $slug        is slug of package to return.
     * @param  int    $ratingPk    is value used to uniquely identify rating.
     * @param  string $verbose     parameter to provide detail view of hyperlink objects.
     * @return array rating details.
     */
    function getRatingDetails($accountSlug, $slug, $ratingPk, $verbose)
    {
        $apiInstance = $this->packageInstance;
        try {
            $sessionKey = $this->sessionKey;
            $result      = $apiInstance->getRatingDetails($accountSlug, $slug, $ratingPk, $sessionKey, $verbose);
            $result = $result->__toString();
            $result = json_decode($result, true);
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }

        return $result;
    }
    
    /**
     * Update rating of package.
     *
     * @param string $accountSlug is user account name.
     * @param string $slug        is slug of package to return.
     * @param int    $ratingPk    is value used to uniquely identify rating.
     * @param array  $body        request data used to update rating.
     */
    function updateRating($accountSlug, $slug, $ratingPk, $body)
    {
        $apiInstance = $this->packageInstance;
        $sessionKey  = $this->sessionKey;
        $body         = new \Swagger\Client\Model\AddRating($body); // \Swagger\Client\Model\AddRating | request data used to update rating.        
        try {
            $apiInstance->updateRating($accountSlug, $slug, $ratingPk, $sessionKey, $body);
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }
    }
    
    /**
     * Get list of licenses associated with package.
     *
     * @param  string $accountslug is user account name.
     * @param  string $slug        is the current user account slug.
     * @param  string $label       is label for the version.
     * @return array license details.
     */
    function getLicenses($accountSlug, $slug, $label)
    {
        $apiInstance = $this->packageInstance;
        $sessionKey  = $this->sessionKey;        
        try {
            $result = $apiInstance->getLicenses($accountSlug, $slug, $label, $sessionKey);
            $result = $result->__toString();
            $result = json_decode($result, true);
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }

        return $result;
    }
    
    /**
     * delete a package license.
     *
     * @param string $accountslug is user account name.
     * @param string $slug        is the current user account slug.
     * @param string $label       is label for the version.
     * @param string $name        is name of the license.
     */
    function removeLicense($accountSlug, $slug, $label, $name)
    {
        $apiInstance = $this->packageInstance;
        $sessionKey  = $this->sessionKey;        
        try {
            $apiInstance->removeLicense($accountSlug, $slug, $label, $name, $sessionKey);
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }
    }
    
    /**
     * Update price of package license.
     *
     * @param string $accountslug is user account name.
     * @param string $slug        is the current user account slug.
     * @param string $label       is label for the version.
     * @param string $name        is name of the license.
     * @param array  $body        is request data used to update license price.
     */
    function updateLicense($accountSlug, $slug, $label, $name, $body)
    {
        $apiInstance = $this->packageInstance;
        $sessionKey  = $this->sessionKey;
        $body         = new \Swagger\Client\Model\UpdateLicense($body); // \Swagger\Client\Model\UpdateLicense | request data used to update license price.        
        try {
            $apiInstance->updateLicense($accountSlug, $slug, $label, $name, $sessionKey, $body);
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }
    }
    
    /**
     * API to acquire package between accounts.
     *
     * @param  string $accountslug is user account name.
     * @param  string $slug        is the current user account slug.
     * @param  array  $body        is request data used to update license price.
     * @return array package object.
     */
    function acquirePackage($accountSlug, $slug, $body)
    {
        $apiInstance = $this->packageInstance;
        $sessionKey  = $this->sessionKey;        
        try {
            $result = $apiInstance->acquirePackage($accountSlug, $slug, $sessionKey, $body);
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject(), $e->getCode());
        }

        return $result;
    }


    /**
     * API to accept package.
     * To share,transfer, get, buy, extend and clone a package.
     * @param string $accountSlug
     * @param string $slug
     * @param array $body
     * @return type
     * @throws Exception
     */
    function acceptAcquisition($accountSlug, $slug, $body) {
         $apiInstance = $this->packageInstance;
        $sessionKey  = $this->sessionKey;        
        try {
            $result = $apiInstance->acceptAcquisition($accountSlug, $slug, $sessionKey, $body);
            $result = json_decode($result, true);
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject(), $e->getCode());
        }

        return $result;
        
    }

    /**
     * API to reject shared package.
     *
     * @param  string $accountslug is user account name.
     * @param  string $slug        is the current user account slug.
     * @param  string $body        is request data used to update license price.
     */
    function rejectAcquisitionInvite($accountSlug, $slug, $body)
    {
        $apiInstance = $this->packageInstance;
        $sessionKey  = $this->sessionKey;
        
        try {
            $apiInstance->rejectAcquisitionInvite($accountSlug, $slug, $sessionKey, $body);
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }
    }

    /**
     * Detail list of those accounts the package is shared or invited for accept share package.
     *
     * @param  string $accountslug is user account name.
     * @param  string $slug        is the current user account slug.
     * @param  array  $body        is request data used to update license price.
     * @return array package object.
     */
    function getAcquiredPackageAccountList($accountSlug, $slug, $share)
    {
        $apiInstance = $this->packageInstance;
        $sessionKey  = $this->sessionKey;
        try {
            $result = $apiInstance->getAcquiredPackageAccountList($accountSlug, $slug, $sessionKey, $share);
            $result = $result->__toString();
            $result = json_decode($result, true);
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }

        return $result;
    }

    /**
     * Search accounts by starting keywords of account name.
     *
     * @param  string $accountslug is user account name.
     * @return array user object.
     */
    function searchAccounts($accountSlug)
    {
        $apiInstance = $this->accountInstance;
        $sessionKey  = $this->sessionKey;        
        try {
            $result = $apiInstance->searchAccounts($sessionKey, $accountSlug);
            $result = $result->__toString();
            $result = json_decode($result, true);
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }

        return $result;
    }
    
    /**
     * Revoke shared pakage from user.
     *
     * @param  string $accountslug is user account name.
     * @param  string $slug        is the current user account slug.
     * @param  array  $body        is request data used to update license price.
     * @return array package object.
     */
    function revokeAcquiredPackage($accountSlug, $slug, $body)
    {
        $apiInstance = $this->packageInstance;
        $sessionKey  = $this->sessionKey;        
        try {
            $result = $apiInstance->revokeAcquiredPackage($accountSlug, $slug, $sessionKey, $body);
            $result = json_decode($result, true);
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }

        return $result;
    }

    /**
     * Create a snapshot of a package.
     *
     * @param  string $accountslug is user account name.
     * @param  string $slug        is the current user account slug.
     * @param  array  $body        is request data used to update license price.
     * @return array package object.
     */
    function snapshot($accountSlug, $slug, $body)
    {
        $apiInstance = $this->packageInstance;
        $sessionKey  = $this->sessionKey;        
        try {
            $result = $apiInstance->snapshot($accountSlug, $slug, $sessionKey, $body);
            $result = $result->__toString();
            $result = json_decode($result, true);
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject(), $e->getCode());
        }

        return $result;
    }
    
    /**
     * Get current user account.
     *
     * @param  string           $slug    is the current user account slug.
     * @param  string[optional] $verbose parameter to provide detail view of hyperlink objects.
     * @return array user account details.
     */
    function getCurrentAccount($slug, $verbose)
    {
        $apiInstance = $this->accountInstance;
        $sessionKey  = $this->sessionKey;        
        try {
            $result = $apiInstance->getCurrentAccount($slug, $sessionKey, $verbose);
            $result = json_decode($result, true);
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }

        return $result;
    }
    
    /**
     * Get list of user's account pending users.
     *
     * @param  string           $slug    is the current user account slug.
     * @param  string[optional] $verbose parameter to provide detail view of hyperlink objects.
     * @return array pending user account details.
     */
    function getPendingUsers($slug, $verbose)
    {
        $apiInstance = $this->accountInstance;
        $sessionKey  = $this->sessionKey;
        try {
            $result = $apiInstance->getPendingUsers($slug, $sessionkey, $verbose);
            $result = $result->__toString();
            $result = json_decode($result, true);
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }

        return $result;
    }
    
    /**
     * Invite user to join account.
     *
     * @param string $slug is the current user account slug.
     * @param array  $body the content of the request used to add admin into user account.
     */
    function inviteUser($slug, $body)
    {
        $apiInstance = $this->accountInstance;
        $sessionKey  = $this->sessionKey;        
        try {
            $apiInstance->inviteUser($slug, $sessionkey, $body);
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }
    }
    
    /**
     * Delete a pending user from user account.
     *
     * @param string $slug is the current user account slug.
     * @param array  $body the content of the request used to add admin into user account.
     */
    function removePendingUser($email, $slug)
    {
        $apiInstance = $this->accountInstance;
        $sessionKey  = $this->sessionKey;        
        try {
            $apiInstance->removePendingUser($pk, $slug, $sessionkey);
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }
    }
    
    /**
     * Update User Account.
     *
     * @param string $slug is the current user account slug.
     * @param array  $body request data used to update user account.
     */
    function updateAccount($slug, $body)
    {
        $apiInstance = $this->accountInstance;
        $sessionKey  = $this->sessionKey;
        $body         = new \Swagger\Client\Model\UpdateAccount($body);        
        try {
            $apiInstance->updateAccount($slug, $body, $sessionKey);
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }
    }

    /**
     * Resend Email Verification link to verifiy account if already not verified.
     *
     * @param array  $body request data used to update user account.
     */
    function resendAccountVerificationEmail($body)
    {
        $apiInstance = $this->accountInstance;
        $sessionKey  = $this->sessionKey;
        $body = new \Swagger\Client\Model\ResendAccountVerificationLink($body);
        try {
            $apiInstance->resendAccountVerificationEmail($sessionKey, $body);
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }
    }
    
    /**
     * Search Packages.
     *
     * @param string[optional] $filterBy filter by parameters.
     * @param string[optional] $orderBy  to order package display.
     */
    function search($filterBy, $orderBy)
    {
        $apiInstance = $this->packageInstance;
        try {
            $result = $apiInstance->search($filterBy, $orderBy);
            $result = $result->__toString();
            $result = json_decode($result, true);
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }

        return $result;
    }
    
    /**
     * Update package draft.
     *
     * @param  string $accountslug is user account name.
     * @param  string $slug        is the current user account slug.
     * @param  string $packageHash is package Hash.
     * @param  array  $body        is request data used to update package draft.
     * @return array package draft.
     */
    function updateDraft($accountSlug, $slug, $packageHash, $body)
    {
        $apiInstance = $this->packageInstance;        
        $body       = array('name' => $slug, 'hash_as_token' => $packageHash, 'documents' => $body);
        $sessionkey = $this->sessionKey;        
        try {
            $result = $apiInstance->updateDraft($accountSlug, $slug, $sessionkey, $body);
            $result = $result->__toString();
            $result = json_decode($result, true);
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }

        return $result;
    }
    
    /**
     * Get package draft.
     *
     * @param  string $accountSlug is the user account slug.
     * @param  string $slug        is the slug of package to return.
     * @return array  package draft.
     */
    function getDraft($accountSlug, $slug)
    {
        $apiInstance = $this->packageInstance;
        $sessionKey  = $this->sessionKey;        
        try {
            $result = $apiInstance->getDraft($accountSlug, $slug, $sessionKey);
            $result = $result->__toString();
            $result = json_decode($result, true);
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }

        return $result;
    }
    
    /**
     * Create package draft.
     *
     * @param  string $accountSlug is the user account slug.
     * @param  string $slug        is the slug of package to return.
     * @param  string $packageHash is package Hash.
     * @param  array  $body        is request data used to update package draft.
     * @return array  package draft.
     */
    function createDraft($accountSlug, $slug, $packageHash, $body)
    {
        $apiInstance = $this->packageInstance;
        $sessionKey  = $this->sessionKey;
        $body         = array('name' => $slug, 'hash_as_token' => $packageHash, 'documents' => $body); 
        try {
            $result = $apiInstance->createDraft($accountSlug, $slug, $sessionKey, $body);
            $result = $result->__toString();
            $result = json_decode($result, true);
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }

        return $result;
    }
    
    /**
     * Upload a media file to package's draft.
     *
     * @param  string $accountSlug   is the user account slug.
     * @param  string $slug          is the slug of package to return.
     * @param  string $packageHash   is image hash.
     * @param  string $name          is file name.
     * @param  object $splFileObject is image file object
     * @return array  package draft.
     */
    function uploadMedia($accountSlug, $slug, $hash, $name, $token, $splFileObject)
    {
        $apiInstance = $this->packageInstance;
        $sessionkey   = $this->sessionKey;
        try {
            $result = $apiInstance->uploadMedia($accountSlug, $slug, $sessionkey, $hash, $name, $token, $splFileObject);
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }

        return $result;
    }
    
    /**
     * Create a new package.
     *
     * @param  array            $body    is request data used to update package draft.
     * @param  string[optional] $verbose parameter to provide detail view of hyperlink objects.
     * @return array  package draft.
     */
    function createPackage($body, $verbose)
    {
        $apiInstance = $this->packageInstance;
        $sessionKey  = $this->sessionKey;        
        try {
            $result = $apiInstance->createPackage($sessionKey, $body, $verbose);
            $result = $result->__toString();
            $result = json_decode($result, true);
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }

        return $result;
    }
    
    /**
     * Create a new package.
     *
     * @param  string           $accountSlug is the user account slug.
     * @param  string           $slug        is the slug of package to return.
     * @param  string[optional] $verbose     parameter to provide detail view of hyperlink objects.
     * @return array  package draft.
     */
    function getCanonical($accountSlug, $slug, $verbose)
    {
        $apiInstance = $this->packageInstance;
        $sessionKey  = $this->sessionKey;        
        try {
            $result = $apiInstance->getCanonical($accountSlug, $slug, $sessionKey, $verbose);
            $result = $result->__toString();
            $result = json_decode($result, true);
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }

        return $result;
    }
    
    /**
     * Download Canonical version of package.
     *
     * @param  string $accountSlug is the user account slug.
     * @param  string $slug        is the slug of package to return.
     * @return array  package object.
     */
    function downloadCanonical($accountSlug, $slug)
    {
        $apiInstance = $this->packageInstance;
        $sessionKey  = $this->sessionKey;        
        try {
            $result = $apiInstance->downloadCanonical($accountSlug, $slug, $sessionKey);
            $result = json_encode($result);
            $result = json_decode($result, true);
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }

        return $result;        
    }
    
    /**
     * Return package dependencies.
     *
     * @param  string           $accountSlug is the user account slug.
     * @param  string           $slug        is the slug of package to return.
     * @param  string[optional] $verbose     parameter to provide detail view of hyperlink objects.
     * @return array  package dependencies.
     */
    function getDependencies($accountSlug, $slug, $verbose)
    {
        $apiInstance = $this->packageInstance;
        $sessionKey  = $this->sessionKey;        
        try {
            $result = $apiInstance->getDependencies($accountSlug, $slug, $sessionKey, $verbose);
            $result = $result->__toString();
            $result = json_decode($result, true);
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }

        return $result;
    }
    
    /**
     * Access draft documents and directory.
     *
     * @param  string $accountSlug is the user account slug.
     * @param  string $slug        is the slug of package to return.
     * @param  string $name        path of the directory. Ex. - meta/config.css
     * @return array draft documents.
     */
    function getDetailDocument($accountSlug, $slug, $name)
    {        
        $apiInstance = $this->packageInstance;
        $sessionkey   = $this->sessionKey;        
        try {
            $result = $apiInstance->getDetailDocument($accountSlug, $slug, $name, $sessionkey);
            $result = json_encode($result);
            $result = json_decode($result, true);
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }

        return $result;
    }
    
    /**
     * Get version data of a package in details.
     *
     * @param  string $accountSlug is the user account slug.
     * @param  string $slug        is the slug of package to return.
     * @param  string $label       value used to uniquely identify version.
     * @return array package version detail.
     */
    function getVersionDetails($accountSlug, $slug, $label)
    {
        $apiInstance = $this->packageInstance;
        $sessionKey  = $this->sessionKey;
        try {
            $result = $apiInstance->getVersionDetails($accountSlug, $slug, $label, $sessionKey);
            $result = $result->__toString();
            $result = json_decode($result, true);
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }

        return $result;
    }
    
    /**
     * Update package's version as canonical version.
     *
     * @param string $accountSlug is the user account slug.
     * @param string $slug        is the slug of package to return.
     * @param string $label       value used to uniquely identify version.
     * @param array  $body        is to request data used to update package version.
     */
    function canonicalSet($accountSlug, $slug, $body)
    {
        $apiInstance = $this->packageInstance;
        $sessionKey  = $this->sessionKey;
        try {
            $apiInstance->canonicalSet($accountSlug, $slug, $sessionKey, $body);
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }
    }
    
    /**
     * Update Package.
     *
     * @param string $accountSlug is the user account slug.
     * @param string $slug        is the slug of package to return.
     * @param array  $body        is to request data used to update package.
     */
    function updatePackage($accountSlug, $slug, $body)
    {
        $apiInstance = $this->packageInstance;
        $sessionKey  = $this->sessionKey;        
        try {
            $apiInstance->updatePackage($accountSlug, $slug, $sessionKey, $body);
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }
    }
    
    /**
     * Create a new user account.
     *
     * @param  array            $body    is the the content of the request used to create user account.
     * @param  string[optional] $verbose parameter to provide detail view of hyperlink objects.
     * @return array user details.
     */
    function createAccount($body, $verbose)
    {
        $apiInstance = $this->accountInstance;
        try {
            $result = $apiInstance->createAccount($body, $verbose);
            $result = $result->__toString();
            $result = json_decode($result, true);
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }

        return $result;        
    }
    
    /**
     * Detail View of user account.
     *
     * @param  string           $slug    is the slug of account to return.
     * @param  string[optional] $verbose parameter to provide detail view of hyperlink objects.
     * @return array account details.
     */
    function getAccountDetails($slug, $verbose)
    {
        $apiInstance = $this->accountInstance;
        $sessionKey  = $this->sessionKey;        
        try {
            $result = $apiInstance->getAccountDetails($slug, $sessionKey, $verbose);
            $result = $result->__toString();
            $result = json_decode($result, true);
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }

        return $result;
    }

    /**
     * Resend the inivite of shared package.
     *
     * @param  string $accountSlug is account slug.
     * @param  string $slug is package name.
     * @param  array  $body is request body.
     * @return array package details.
     */
    function resendInvite($accountSlug, $slug, $body)
    {
        $apiInstance = $this->packageInstance;
        $sessionKey  = $this->sessionKey;
       
        $body = new \Swagger\Client\Model\ResendInvite($body);
        
        try {
            $result = $apiInstance->resendInvite($accountSlug, $slug, $sessionKey, $body);
            $result = json_decode($result, true);
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }

        return $result;
    }
    
    /**
     * Returns all accounts from the system that the user has access to.
     *
     * @param  string           $slug    is the slug of account to return.
     * @param  string[optional] $verbose parameter to provide detail view of hyperlink objects.
     * @return array account list.
     */
    function getAccountList($slug, $verbose)
    {
        $apiInstance = $this->accountInstance;
        $sessionKey  = $this->sessionKey;        
        try {
            $result = $apiInstance->getAccountList($sessionKey, $slug, $verbose);
            $result = $result->__toString();
            $result = json_decode($result, true);
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }

        return $result;
    }
    
    /**
     * Reset password of user account.
     *
     * @param array $body is the the content of the request used to create user account.
     */
    function resetPassword($body)
    {
        $apiInstance = $this->accountInstance;        
        try {
            $apiInstance->resetPassword($body);
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }
    }
    
    /**
     * Returns all accounts from the system that the user has access to.
     *
     * @param string $slug is the slug of account to return.
     * @param array  $body is the the content of the request used to create user account.
     */
    function setCurrentAccount($slug, $body)
    {
        $apiInstance = $this->accountInstance;
        $sessionKey  = $this->sessionKey;
        try {
            $apiInstance->setCurrentAccount($slug, $sessionKey, $body);
        }
        catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }
    }

    /**
     * Return the list of package version.
     *
     * @param  string $accountSlug is the user account slug.
     * @param  string $slug        is the slug of package to return.
     * @return array version detail.
     */
    function getVersions($accountSlug, $slug)
    {
        $apiInstance = $this->packageInstance;
        $sessionKey  = $this->sessionKey;
        try {
            $result = $apiInstance->getVersions($accountSlug, $slug, $sessionKey);
            $result = $result->__toString();
            $result = json_decode($result, true);
        } catch (\Exception $e) {
            throw new \Exception($e->getResponseObject());
        }

        return $result;
    }    

    /**
     * Get initial packages from hub.
     *
     * @return array acquisition package list.
     */
    function getPackageInitial() 
    {
        $apiInstance = $this->packageInstance;     
        try {
            $result = $apiInstance->getPackageInitial();
            $result = $result->__toString();
            $result = json_decode($result, true);
        } catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }

        return $result;
    }

    /**
     * Get initial packages from hub.
     *
     * @return array acquisition package list.
     */
    function sharedPackageListGet($account_slug, $slug) 
    {
        $api_instance = new \Swagger\Client\Api\LibraryApi();
        $sessionKey  = $this->sessionKey;
        try { 
            $result = $api_instance->sharedPackageListGet($account_slug, $slug, $sessionkey);
        } catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }

        return $result;
    }

    /**
     * Get package publications
     * 
     * @param string $accountSlug
     * @param string $slug
     * @return array package publications.
     * @throws \Exception
     */
    function getPublication($accountSlug, $slug) 
    {
        $apiInstance = $this->packageInstance; 
        $sessionKey  = $this->sessionKey;
        try { 
            $result = $apiInstance->getPublication($accountSlug, $slug, $sessionKey);
            $result = $result->__toString();
            $result = json_decode($result, true);
        } catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }

        return $result;
    }
    
    
    /**
     * Update package publications
     * 
     * @param string $accountSlug
     * @param string $slug
     * @param string $sessionKey
     * @param array $body
     * @return array package publications.
     * @throws \Exception
     */
    function updatePublication($accountSlug, $slug, $commit, $body) {
        $apiInstance = $this->packageInstance; 
        $sessionKey  = $this->sessionKey;
        try { 
            $result = $apiInstance->updatePublication($accountSlug, $slug, $body, $sessionKey, $commit);
        } catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }

        return $result;        
    }
    
    
    /**
     * Validate package publications
     * 
     * @param string $accountSlug
     * @param string $slug
     * @param string $sessionKey
     * @param array $body
     * @return array package publications.
     * @throws \Exception
     * @todo check return type after React test
     */
    function validatePublication($accountSlug, $slug, $body) {
        $apiInstance = $this->packageInstance;
        $sessionKey  = $this->sessionKey;
        try { 
            $result = $apiInstance->validatePublication($accountSlug, $slug, $sessionKey, $body);
        } catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }

        return $result;
    }
    
    
    /**
     * Get package version bundles
     * 
     * @param string $accountSlug
     * @param string $slug
     * @return array package version bundles
     * @throws \Exception
     */
    function getVersionBundles($accountSlug, $slug) {
        $apiInstance = $this->packageInstance;
        $sessionKey  = $this->sessionKey;
        try { 
            $result = $apiInstance->getVersionBundles($accountSlug, $slug, $sessionKey);
            $result = $result->__toString();
            $result = json_decode($result, true);
        } catch (ApiException $e) {
            throw new \Exception($e->getResponseObject());
        }

        return $result;
    }
}
