<?php

/**
 * Blended REST Accounts Controller 
 *
 * @class    Blended
 * @author   Blended
 * @package  Blended/Rest/Accounts
 * @version  1.0.0
 */
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

class BD_REST_Accounts_Controller extends BD_REST_Controller {

    protected $rest_base = 'accounts';

    public function register_routes() {
        register_rest_route(
                $this->namespace, '/' . $this->rest_base, array(
            array(
                'methods' => WP_REST_Server::READABLE,
                'callback' => array($this, 'get_localaccountDetails')
            ),
            array(
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => array($this, 'createAccount')
            ))
        );

        register_rest_route(
                $this->namespace, '/' . $this->rest_base . '/(?P<slug>[a-z0-9_\-]+)', array(
            array(
                'methods' => WP_REST_Server::READABLE,
                'callback' => array($this, 'get_getAccountDetails'),
            ),
            array(
                'methods' => WP_REST_Server::EDITABLE,
                'callback' => array($this, 'updateAccount'),
            ))
        );
        
        register_rest_route(
                $this->namespace, '/' . $this->rest_base . '/(?P<slug>[a-z0-9_\-]+)/pendings', array(
            array(
                'methods' => WP_REST_Server::READABLE,
                'callback' => array($this, 'getPendingUsers'),
            ),
            array(
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => array($this, 'inviteUser'),
            ))
        );
        
        register_rest_route(
                $this->namespace, '/' . $this->rest_base . '/(?P<slug>[a-z0-9_\-]+)/pendings/(?P<email>[a-z0-9.%#!$&/|*_-]+)', array(
            array(
                'methods' => WP_REST_Server::DELETABLE,
                'callback' => array($this, 'removePendingUser'),
            ))
        );

        register_rest_route(
                $this->namespace, '/' . $this->rest_base . '/(?P<slug>[a-z0-9_\-]+)/accounts', array(
            array(
                'methods' => WP_REST_Server::READABLE,
                'callback' => array($this, 'get_getAccountList'),
            ))
        );
        
        register_rest_route(
                $this->namespace, '/' . $this->rest_base . '/(?P<slug>[a-z0-9_\-]+)/admins', array(
            array(
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => array($this, 'addAccountAdmin'),
            ))
        );

        register_rest_route(
                $this->namespace, '/' . $this->rest_base . '/resend/emailverification', array(
            array(
                'methods' => WP_REST_Server::EDITABLE,
                'callback' => array($this, 'resendAccountVerificationEmail'),
            ))
        );
        
        register_rest_route(
                $this->namespace, '/' . $this->rest_base . '/search/(?P<accountSlug>[a-z0-9_\-]+)', array(
            array(
                'methods' => WP_REST_Server::READABLE,
                'callback' => array($this, 'searchAccounts'),
            ))
        );
        
        register_rest_route(
                $this->namespace, '/' . $this->rest_base . '/(?P<slug>[a-z0-9_\-]+)/current', array(
            array(
                'methods' => WP_REST_Server::READABLE,
                'callback' => array($this, 'get_getCurrentAccount'),
            ))
        );

        register_rest_route(
                $this->namespace, '/' . $this->rest_base . '/(?P<userSlug>[a-z0-9_\-]+)/current', array(
            array(
                'methods' => WP_REST_Server::EDITABLE,
                'callback' => array($this, 'put_setCurrentAccount'),
            ),
                )
        );
        
        register_rest_route(
                $this->namespace, '/' . 'reset', array(
            array(
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => array($this, 'resetPassword'),
            ),
                )
        );
    }

    /**
     * GET: Get local account details
     * @param object $request
     * @return WP_Error|WP_REST_Response
     */
    public function get_localaccountDetails($request) {
        try {
            $wpSession = BD()->api->authentication->session_key;
            BD()->network_key = BD()->api->authentication->blended_session_setter($wpSession);
            $userDetails = $this->itinerary_route->get_localaccountDetails();
            $response = rest_ensure_response($userDetails);
            $response->set_status(200);
            return $response;
        } catch (Exception $error) {
            if (isset($error['message'])) {
                return new WP_Error($error['status_code'], $error['message'], "");
            }
        }
    }

    /**
     * GET: Get account details
     * @param object $request
     * @return WP_Error|WP_REST_Response
     */
    public function get_getAccountDetails($request) {
        try {
            $slug = strtolower($request['slug']);
            $wpSession = BD()->api->authentication->session_key;
            BD()->network_key = BD()->api->authentication->blended_session_setter($wpSession);
            $verbose = true;
            $accountDetail = $this->itinerary_route->get_getAccountDetails($slug, $verbose, $wpSession);
            $response = rest_ensure_response($accountDetail);
            $response->set_status(200);
            return $response;
        } catch (Exception $e) {
            $error = json_decode($e->getMessage(), true);
            if (isset($error['status_code']) && isset($error['message'])) {
                return new WP_Error($error['status_code'], $error['message'], "");
            } else {
                return new WP_Error("500", $e->getMessage(), "");
            }
        }
    }

    /**
     * GET: Get account list
     * @param object $request
     * @return WP_Error|WP_REST_Response
     */
    public function get_getAccountList($request) {
        try {
            $slug = strtolower($request['slug']);
            $wpSession = BD()->api->authentication->session_key;
            BD()->network_key = BD()->api->authentication->blended_session_setter($wpSession);
            $verbose = true;
            $accountList = $this->itinerary_route->get_getAccountList($slug, $verbose, $wpSession);
            $response = rest_ensure_response($accountList);
            $response->set_status(200);
            return $response;
        } catch (Exception $e) {
            $error = json_decode($e->getMessage(), true);
            return new WP_Error($error['status_code'], $error['message'], "");
        }
    }

    /**
     * GET: Get current account
     * @param object $request
     * @return WP_Error|WP_REST_Response
     */
    public function get_getCurrentAccount($request) {
        try {
            $slug = strtolower($request['slug']);
            $wpSession = BD()->api->authentication->session_key;
            BD()->network_key = BD()->api->authentication->blended_session_setter($wpSession);
            $verbose = true;
            $accountList = $this->itinerary_route->get_getCurrentAccount($slug, $verbose, $wpSession);
            $response = rest_ensure_response($accountList);
            $response->set_status(200);
            return $response;
        } catch (Exception $e) {
            $error = json_decode($e->getMessage(), true);
            return new WP_Error($error['status_code'], $error['message'], "");
        }
    }
    
    /**
     * PUT: Save current account
     * @param object $request
     * @return WP_Error|WP_REST_Response
     */
    public function put_setCurrentAccount($request) {
        try {
            $body = $this->get_body($request);
            $wpSession = BD()->api->authentication->session_key;
            $slug = strtolower($request['userSlug']);
            BD()->network_key = BD()->api->authentication->blended_session_setter($wpSession);
            $this->itinerary_route->put_setCurrentAccount($slug, $body, $wpSession);
        } catch (Exception $e) {
            $error = json_decode($e->getMessage(), true);
            return new WP_Error($error['status_code'], $error['message'], "");
        }
    }
    
    /**
     * POST: Create a new user account
     * @param object $request
     * @return WP_Error|WP_REST_Response
     */
    public function createAccount($request) {
        try {
            $verbose = true;
            $body = $this->get_body($request);
            $userDetail = $this->itinerary_route->post_createAccount($body, $verbose);
            $response = rest_ensure_response($userDetail);
            $response->set_status(200);
            return $response;
        } catch (Exception $e) {
            $error = json_decode($e->getMessage(), true);
            return new WP_Error($error['status_code'], $error['message'], "");
        }
    }
    
    /**
     * PUT: Update User Account
     * @param object $request
     * @return WP_Error|WP_REST_Response
     */
    public function updateAccount($request) {
        try {
            $body = $this->get_body($request);
            $wpSession = BD()->api->authentication->session_key;
            $slug = strtolower($request['slug']);
            BD()->network_key = BD()->api->authentication->blended_session_setter($wpSession);
            if (isset($body['activationSolution'])) {
                $body['activation_solution'] = $body['activationSolution'];
                unset($body['activationSolution']);
            }
            $updateAccount = $this->itinerary_route->put_updateAccount($slug, $body, $wpSession);
            $response = rest_ensure_response($updateAccount);
            $response->set_status(200);
            return $response;
        } catch (Exception $e) {
            $error = json_decode($e->getMessage(), true);
            return new WP_Error($error['status_code'], $error['message'], "");
        }
    }
    
    /**
     * GET: Get list of user's account pending users
     * @param object $request
     * @return WP_Error|WP_REST_Response
     */
    public function getPendingUsers($request) {
        try {
            $slug = strtolower($request['slug']);
            $wpSession = BD()->api->authentication->session_key;
            BD()->network_key = BD()->api->authentication->blended_session_setter($wpSession);
            $verbose = true;
            $pendingUser = $this->itinerary_route->get_getPendingUsers($slug, $verbose, $wpSession);
            $response = rest_ensure_response($pendingUser);
            $response->set_status(200);
            return $response;
        } catch (Exception $e) {
            $error = json_decode($e->getMessage(), true);
            return new WP_Error($error['status_code'], $error['message'], "");
        }
    }
    
    /**
     * POST: Invite user to join account
     * @param object $request
     * @return WP_Error|WP_REST_Response
     */
    public function inviteUser($request) {
        try {
            $wpSession = BD()->api->authentication->session_key;
            BD()->api->authentication->blended_session_setter($wpSession);
            $body = $this->get_body($request);
            $slug = strtolower($request['slug']);
            $inviteUser = $this->itinerary_route->post_inviteUser($slug, $body, $wpSession);
            $response = rest_ensure_response($inviteUser);
            $response->set_status(200);
            return $response;
        } catch (Exception $e) {
            $error = json_decode($e->getMessage(), true);
            return new WP_Error($error['status_code'], $error['message'], "");
        }
    }
    
    /**
     * DELETE: Delete a pending user from user account
     * @param object $request
     * @return WP_Error|WP_REST_Response
     */
    public function removePendingUser($request) {
        try {
            $wpSession = BD()->api->authentication->session_key;
            BD()->api->authentication->blended_session_setter($wpSession);
            $email = $request['email'];
            $slug = strtolower($request['slug']);
            $remove = $this->itinerary_route->delete_removePendingUser($slug, $email, $wpSession);
            $response = rest_ensure_response($remove);
            $response->set_status(200);
            return $response;
        } catch (Exception $e) {
            $error = json_decode($e->getMessage(), true);
            return new WP_Error($error['status_code'], $error['message'], "");
        }
    }
    
    /**
     * POST: Add a admin user into account
     * @param object $request
     * @return WP_Error|WP_REST_Response
     */
    public function addAccountAdmin($request) {
        try {
            $wpSession = BD()->api->authentication->session_key;
            BD()->api->authentication->blended_session_setter($wpSession);
            $body = $this->get_body($request);
            $slug = $body['slug'];
            $addAccount = $this->itinerary_route->post_addAccountAdmin($slug, $body, $wpSession);
            $response = rest_ensure_response($addAccount);
            $response->set_status(200);
            return $response;
        } catch (Exception $e) {
            $error = json_decode($e->getMessage(), true);
            return new WP_Error($error['status_code'], $error['message'], "");
        }
    }
    
    /**
     * PUT: Resend Email Verification link to verifiy account if already not verified
     * @param object $request
     * @return WP_Error|WP_REST_Response
     */
    public function resendAccountVerificationEmail($request) {
        try {
            $body = $this->get_body($request);
            $wpSession = BD()->api->authentication->session_key;
            BD()->api->authentication->blended_session_setter($wpSession);
            $resendAccount = $this->itinerary_route->put_resendAccountVerificationEmail($wpSession, $body);
            $response = rest_ensure_response($resendAccount);
            $response->set_status(200);
            return $response;
        } catch (Exception $e) {
            $error = json_decode($e->getMessage(), true);
            if (isset($error['status_code']) && $error['message']) {
                return new WP_Error($error['status_code'], $error['message'], "");
            } else if (isset($error['errors'])) {
                return new WP_Error($error['errors'], $error['errors'], "");
            } else {
                return json_encode($e->getMessage());
            }
        }
    }
    
    /**
     * GET: Search accounts by starting keywords of account name
     * @param object $request
     * @return WP_Error|WP_REST_Response
     */
    public function searchAccounts($request) {
        try {
            $wpSession = BD()->api->authentication->session_key;
            BD()->network_key = BD()->api->authentication->blended_session_setter($wpSession);
            $accountSlug     = strtolower($request['accountSlug']);
            $accountDetail = $this->itinerary_route->get_searchAccounts($accountSlug, $wpSession);
            $response = rest_ensure_response($accountDetail);
            $response->set_status(200);
            return $response;
        } catch (Exception $e) {
            $error = json_decode($e->getMessage(), true);
            return new WP_Error($error['status_code'], $error['message'], "");
        }
    }
    
    /**
     * POST: Reset password of user account
     * @param object $request
     * @return WP_Error|WP_REST_Response
     */
    public function resetPassword($request) {
        try {
            $wpSession = BD()->api->authentication->session_key;
            BD()->api->authentication->blended_session_setter($wpSession);
            $body = $this->get_body($request);
            $resetPassword = $this->itinerary_route->post_resetPassword($body, $wpSession);
            $response = rest_ensure_response($resetPassword);
            $response->set_status(200);
            return $response;
        } catch (Exception $e) {
            $error = json_decode($e->getMessage(), true);
            return new WP_Error($error['status_code'], $error['message'], "");
        }
    }

}
