<?php

/**
 * Blended REST Packages Acquisitions Controller 
 *
 * @class    Blended
 * @author   Blended
 * @package  Blended/Rest/Packages
 * @version  1.0.0
 */
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

class BD_REST_Packages_Acquisitions_Controller extends BD_REST_Packages_Controller {

    /**
     * Endpoint
     * @var string
     */
    protected $rest_base = 'packages/(?P<accountSlug>[a-z0-9_\-]+)/(?P<slug>[a-z0-9_\-]+)/acquisitions';

    public function register_routes() {
        register_rest_route(
                $this->namespace, '/' . $this->rest_base, array(
            array(
                'methods' => WP_REST_Server::READABLE,
                'callback' => array($this, 'get_items')
            ),
             array(
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => array($this, 'acquirePackage')
            ),
            array(
                'methods' => WP_REST_Server::EDITABLE,
                'callback' => array($this, 'acceptAcquisition')
            ),
            array(
                'methods' => WP_REST_Server::DELETABLE,
                'callback' => array($this, 'revokeAcquiredPackage')
            ),
            )
        );
        
        register_rest_route(
                $this->namespace, '/packages/(?P<accountSlug>[a-z0-9_\-]+)/(?P<slug>[a-z0-9_\-]+)/reject', array(
            array(
                'methods' => WP_REST_Server::EDITABLE,
                'callback' => array($this, 'rejectAcquisitionInvite')
            ))
        );
        
        register_rest_route(
                $this->namespace, '/packages/(?P<accountSlug>[a-z0-9_\-]+)/(?P<slug>[a-z0-9_\-]+)/resend/invite', array(
            array(
                'methods' => WP_REST_Server::EDITABLE,
                'callback' => array($this, 'resendInvite')
            ))
        );
        
        
    }
    
    /**
     * GET: Get acquired package accounts
     * @param object $request
     * @return WP_Error|WP_REST_Response
     */
    public function get_items($request) {
        try {
            $wpSession = BD()->api->authentication->session_key;
            BD()->api->authentication->blended_session_setter($wpSession);
            $accountSlug = strtolower($request['accountSlug']);
            $slug = strtolower($request['slug']);
            $share = $request['share'];
            $packages = $this->itinerary_route->get_getAcquiredPackageAccountList($accountSlug, $slug, $wpSession, $share);
            $response = rest_ensure_response($packages);
            $response->set_status(200);
            return $response;
        } catch (Exception $e) {
            $error = json_decode($e->getMessage(), true);
            if (isset($error['status_code']) && $error['message']) {
                return new WP_Error($error['status_code'], $error['message'], "");
            } else if (isset($error['errors'])) {
                return new WP_Error($error['errors'], $error['errors'], "");
            } else {
                return json_encode($e->getMessage());
            }
        }
    }

    /**
     * POST: acquire package between accounts
     * @param object $request
     * @return WP_Error|WP_REST_Response
     */
    public function acquirePackage($request) {
        try {
            $wpSession = BD()->api->authentication->session_key;
            BD()->api->authentication->blended_session_setter($wpSession);
            $body = $this->get_body($request);
            $accountSlug = strtolower($request['accountSlug']);
            $slug = strtolower($request['slug']);
            $acquireDetail = $this->itinerary_route->post_acquirePackage($accountSlug, $slug, $body, $wpSession);
            if (empty($acquireDetail)) {
                $acquireDetail = array();
            }
            $response = rest_ensure_response($acquireDetail);
            $response->set_status(200);
            return $response;
        } catch (Exception $e) {
            $code = $e->getCode();
            $error = json_decode($e->getMessage(), true);
            $message = (isset($error['message']) ? $error['message'] : "");
            $status_code = (isset($error['status_code']) ? $error['status_code'] : "");
            $draft_dependency = isset($error['draft_dependency']) && !empty($error['draft_dependency']) ? array("draft_dependency" => $error['draft_dependency']) : "";
            if (!empty($draft_dependency)) {
                return new WP_Error($error['status_code'], $error['message'], $draft_dependency);
            } else if ($code === 500) {
                return new WP_Error($error['status_code'], $error['message'], "");
            }
            $data = prepare_response_data($error, $code);
            return new WP_Error($status_code, $message, $data);
        }
    }

    /**
     * PUT: accept acquisition
     * @param object $request
     * @return WP_Error|WP_REST_Response
     */
    public function acceptAcquisition($request) {
        try {
            $wpSession = BD()->api->authentication->session_key;
            BD()->api->authentication->blended_session_setter($wpSession);
            $body = $this->get_body($request);
            $accountSlug = strtolower($request['accountSlug']);
            $slug = strtolower($request['slug']);
            $packages = $this->itinerary_route->put_acceptAcquisition($accountSlug, $slug, $body, $wpSession);
            $response = rest_ensure_response($packages);
            $response->set_status(200);
            return $response;
        } catch (Exception $e) {
            $error = json_decode($e->getMessage(), true);
            if (isset($error['status_code']) && $error['message']) {
                return new WP_Error($error['status_code'], $error['message'], "");
            } else if (isset($error['errors'])) {
                return new WP_Error($error['errors'], $error['errors'], "");
            } else {
                return json_encode($e->getMessage());
            }
        }
    }
    
    /**
     * DELETE: Revoke shared package from user
     * @param object $request
     * @return WP_Error|WP_REST_Response
     */
    public function revokeAcquiredPackage($request) {
        try {
            $wpSession = BD()->api->authentication->session_key;
            BD()->api->authentication->blended_session_setter($wpSession);
            $body = $this->get_body($request);
            $accountSlug = strtolower($request['accountSlug']);
            $slug = strtolower($request['slug']);
            $packages = $this->itinerary_route->delete_revokeAcquiredPackage($accountSlug, $slug, $body, $wpSession);
            $response = rest_ensure_response($packages);
            $response->set_status(200);
            return $response;
        } catch (Exception $e) {
            $error = json_decode($e->getMessage(), true);
            if (isset($error['status_code']) && $error['message']) {
                return new WP_Error($error['status_code'], $error['message'], "");
            } else if (isset($error['errors'])) {
                return new WP_Error($error['errors'], $error['errors'], "");
            } else {
                return json_encode($e->getMessage());
            }
        }
    }
    
    /**
     * PUT: Reject shared package
     * @param object $request
     * @return WP_Error|WP_REST_Response
     */
    public function rejectAcquisitionInvite($request) {
        try {
            $wpSession = BD()->api->authentication->session_key;
            BD()->api->authentication->blended_session_setter($wpSession);
            $body = $this->get_body($request);
            $accountSlug = strtolower($request['accountSlug']);
            $slug = strtolower($request['slug']);
            $packages = $this->itinerary_route->put_rejectAcquisitionInvite($accountSlug, $slug, $body, $wpSession);
            $response = rest_ensure_response($packages);
            $response->set_status(200);
            return $response;
        } catch (Exception $e) {
            $error = json_decode($e->getMessage(), true);
            if (isset($error['status_code']) && $error['message']) {
                return new WP_Error($error['status_code'], $error['message'], "");
            } else if (isset($error['errors'])) {
                return new WP_Error($error['errors'], $error['errors'], "");
            } else {
                return json_encode($e->getMessage());
            }
        }
    }
    
    /**
     * POST: Resend the invite of shared package
     * @param object $request
     * @return WP_Error|WP_REST_Response
     */
    public function resendInvite($request) {
        try {
            $wpSession = BD()->api->authentication->session_key;
            BD()->api->authentication->blended_session_setter($wpSession);
            $body = $this->get_body($request);
            $accountSlug = strtolower($request['accountSlug']);
            $slug = strtolower($request['slug']);
            $acquireDetail = $this->itinerary_route->post_resendInvite($accountSlug, $slug, $body, $wpSession);
            $response = rest_ensure_response($acquireDetail);
            $response->set_status(200);
            return $response;
        } catch (Exception $e) {
            $error = json_decode($e->getMessage(), true);
            if (isset($error['status_code']) && $error['message']) {
                return new WP_Error($error['status_code'], $error['message'], "");
            } else if (isset($error['errors'])) {
                return new WP_Error($error['errors'], $error['errors'], "");
            } else {
                return json_encode($e->getMessage());
            }
        }
    }

}
