<?php

/**
 * Blended REST Packages Controller 
 *
 * @class    Blended
 * @author   Blended
 * @package  Blended/Rest/Packages
 * @version  1.0.0
 */
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

class BD_REST_Packages_Controller extends BD_REST_Controller {

    /**
     * Endpoint
     * @var string
     */
    protected $rest_base = 'packages';

    public function register_routes() {
        register_rest_route(
                $this->namespace, '/' . $this->rest_base, array(
            array(
                'methods' => WP_REST_Server::READABLE,
                'callback' => array($this, 'get_items')
            ))
        );
        
        register_rest_route(
                $this->namespace, '/' . 'active', array(
            array(
                'methods' => WP_REST_Server::READABLE,
                'callback' => array($this, 'get_active_theme')
            ),
            array(
                'methods' => WP_REST_Server::EDITABLE,
                'callback' => array($this, 'set_active_theme')
            ))
        );
        
        register_rest_route(
                $this->namespace, '/' . $this->rest_base . '/(?P<accountSlug>[a-z0-9_\-]+)/(?P<slug>[a-z0-9_\-]+)', array(
            array(
                'methods' => WP_REST_Server::READABLE,
                'callback' => array($this, 'getPackageDetails')
            ))
        );
        
        register_rest_route(
                $this->namespace, '/' . $this->rest_base . '/(?P<accountSlug>[a-z0-9_\-]+)/(?P<slug>[a-z0-9_\-]+)/currenthash', array(
            array(
                'methods' => WP_REST_Server::READABLE,
                'callback' => array($this, 'getCurrentHash')
            ))
        );
        
        register_rest_route(
                $this->namespace, '/' . $this->rest_base . '/(?P<accountSlug>[a-z0-9_\-]+)/(?P<slug>[a-z0-9_\-]+)/hashes', array(
            array(
                'methods' => WP_REST_Server::READABLE,
                'callback' => array($this, 'getPackageHashes')
            ))
        );
        
        register_rest_route(
                $this->namespace, '/' . $this->rest_base . '/(?P<accountSlug>[a-z0-9_\-]+)/(?P<slug>[a-z0-9_\-]+)/dependencies', array(
            array(
                'methods' => WP_REST_Server::READABLE,
                'callback' => array($this, 'getDependencies')
            ))
        );
        
        register_rest_route(
                $this->namespace, '/' . $this->rest_base . '/(?P<accountSlug>[a-z0-9_\-]+)/(?P<slug>[a-z0-9_\-]+)/ratings/(?P<ratingPk>\d+)', array(
            array(
                'methods' => WP_REST_Server::READABLE,
                'callback' => array($this, 'getRatingDetails')
            ),
            array(
                'methods' => WP_REST_Server::EDITABLE,
                'callback' => array($this, 'updateRating')
            ))
        );
    }
    
    /**
     * GET: Get packages
     * @param object $request
     * @return WP_Error|WP_REST_Response
     */
    public function get_items($request) {
        try {
            $wpSession = BD()->api->authentication->session_key;
            BD()->api->authentication->blended_session_setter($wpSession);
            $share = isset($request['share']) ? $request['share'] : "";
            $transfer_to_me = isset($request['transfer_to_me']) ? $request['transfer_to_me'] : "";
            $transfer_by_me = isset($request['transfer_by_me']) ? $request['transfer_by_me'] : "";
            $snapshot = isset($request['snapshot']) ? $request['snapshot'] : "";
            $purchased = isset($request['purchased']) ? $request['purchased'] : "";
            $dependency = isset($request['dependency']) ? $request['dependency'] : "";
            $published = isset($request['published']) ? $request['published'] : "";
            $owned_by_account = isset($request['owned_by_account']) ? $request['owned_by_account'] : "";
            $packages = $this->itinerary_route->get_sync_client_account($wpSession, $share, $transfer_by_me, $transfer_to_me, $purchased, $snapshot, $dependency, $published, $owned_by_account);
            $response = rest_ensure_response($packages);
            $response->set_status(200);
            return $response;
        } catch (Exception $e) {
            $error = json_decode($e->getMessage(), true);
            return new WP_Error($error['status_code'], $error['message'], "");
        }
    }
    
    /**
     * GET: Get active theme
     * @param object $request
     * @return WP_Error|WP_REST_Response
     */
    public function get_active_theme($request) {
        try {
            $active_theme = $this->itinerary_route->get_active();
            $response = rest_ensure_response($active_theme);
            $response->set_status(200);
            return $response;
        } catch (Exception $e) {
            $error = json_decode($e->getMessage(), true);
            return new WP_Error($error['status_code'], $error['message'], "");
        }
    }
    
    /**
     * PUT: Set active theme
     * @param object $request
     * @return WP_Error|WP_REST_Response
     */
    public function set_active_theme($request) {
        try {
            $wpSession = BD()->api->authentication->session_key;
            BD()->api->authentication->blended_session_setter($wpSession);
            $body = $this->get_body($request);
            $slug = strtolower($body['slug']);
            $activeTheme = $this->itinerary_route->put_active($slug, $wpSession);
            $response = rest_ensure_response($activeTheme);
            $response->set_status(200);
            return $response;
        } catch (Exception $e) {
            $error = json_decode($e->getMessage(), true);
            $data = isset($error['data']) ? $error['data'] : "";
            return new WP_Error($error['status_code'], $error['message'], $data);
        }
    }
    
    /**
     * GET: Get package dependencies
     * @param object $request
     * @return WP_Error|WP_REST_Response
     */
    public function getDependencies($request) {
        try {
            $wpSession = BD()->api->authentication->session_key;
            BD()->api->authentication->blended_session_setter($wpSession);
            $slug = strtolower($request['slug']);
            $accountSlug = strtolower($request['accountSlug']);
            $verbose = true;
            $packageDependencies = $this->itinerary_route->get_getDependencies($accountSlug, $slug, $verbose, $wpSession);
            $response = rest_ensure_response($packageDependencies);
            $response->set_status(200);
            return $response;
        } catch (Exception $e) {
            $error = json_decode($e->getMessage(), true);
            return new WP_Error($error['status_code'], $error['message'], "");
        }
    }
    
    /**
     * GET: Get package draft details
     * @param object $request
     * @return WP_Error|WP_REST_Response
     */
    public function getPackageDetails($request) {
        try {
            $wpSession = BD()->api->authentication->session_key;
            BD()->api->authentication->blended_session_setter($wpSession);
            $slug = strtolower($request['slug']);
            $accountSlug = strtolower($request['accountSlug']);
            $packageDetail = $this->itinerary_route->get_getPackageDetails($accountSlug, $slug, $wpSession);
            $response = rest_ensure_response($packageDetail);
            $response->set_status(200);
            return $response;
        } catch (Exception $e) {
            $error = json_decode($e->getMessage(), true);
            return new WP_Error($error['status_code'], $error['message'], "");
        }
    }
    
    /**
     * GET: Get current package hash of local package
     * @param object $request
     * @return WP_Error|WP_REST_Response
     */
    public function getCurrentHash($request) {
        try {
            $wpSession = BD()->api->authentication->session_key;
            BD()->api->authentication->blended_session_setter($wpSession);
            $slug = strtolower($request['slug']);
            $accountSlug = strtolower($request['accountSlug']);
            $packageDetail = $this->itinerary_route->get_currenthash($accountSlug, $slug, $wpSession);
            $response = rest_ensure_response($packageDetail);
            $response->set_status(200);
            return $response;
        } catch (Exception $e) {
            $error = json_decode($e->getMessage(), true);
            return new WP_Error($error['status_code'], $error['message'], "");
        }
    }
    
    /**
     * GET: Get package all relevant hashes 
     * @param object $request
     * @return WP_Error|WP_REST_Response
     */
    public function getPackageHashes($request) {
        try {
            $wpSession = BD()->api->authentication->session_key;
            BD()->api->authentication->blended_session_setter($wpSession);
            $slug = strtolower($request['slug']);
            $accountSlug = strtolower($request['accountSlug']);
            $hashes = $this->itinerary_route->get_getPackageHashes($accountSlug, $slug, $wpSession);
            $response = rest_ensure_response($hashes);
            $response->set_status(200);
            return $response;
        } catch (Exception $e) {
            $error = json_decode($e->getMessage(), true);
            return new WP_Error($error['status_code'], $error['message'], "");
        }
    }

        /**
     * GET: Get package ratings
     * @param object $request
     * @return WP_Error|WP_REST_Response
     */
    public function getRatingDetails($request) {
        try {
            $wpSession = BD()->api->authentication->session_key;
            BD()->api->authentication->blended_session_setter($wpSession);
            $slug = strtolower($request['slug']);
            $accountSlug = strtolower($request['accountSlug']);
            $ratingPk = (int) $request['ratingPk'];
            $verbose = true;
            $getRating = $this->itinerary_route->get_getRatingDetails($accountSlug, $slug, $ratingPk, $verbose, $wpSession);
            $response = rest_ensure_response($getRating);
            $response->set_status(200);
            return $response;
        } catch (Exception $e) {
            $error = json_decode($e->getMessage(), true);
            return new WP_Error($error['status_code'], $error['message'], "");
        }
    }

    /**
     * PUT: Update rating of a package
     * @param object $request
     * @return WP_Error|WP_REST_Response
     */
    public function updateRating($request) {
        try {
            $wpSession = BD()->api->authentication->session_key;
            BD()->api->authentication->blended_session_setter($wpSession);
            $slug = strtolower($request['slug']);
            $accountSlug = strtolower($request['accountSlug']);
            $ratingPk    = (int) $request['ratingPk'];
            $body = $this->get_body($request);
            $getRating = $this->itinerary_route->put_updateRating($accountSlug, $slug, $ratingPk, $body, $wpSession);
            $response = rest_ensure_response($getRating);
            $response->set_status(200);
            return $response;
        } catch (Exception $e) {
            $error = json_decode($e->getMessage(), true);
            return new WP_Error($error['status_code'], $error['message'], "");
        }
    }

}
