<?php

/**
 * Blended REST Packages Draft Controller 
 *
 * @class    Blended
 * @author   Blended
 * @package  Blended/Rest/Packages
 * @version  1.0.0
 */
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

class BD_REST_Packages_Draft_Controller extends BD_REST_Packages_Controller {

    /**
     * Endpoint
     * @var string
     */
    protected $rest_base = 'packages/(?P<accountSlug>[a-z0-9_\-]+)/(?P<slug>[a-z0-9_\-]+)/draft';

    public function register_routes() {
        register_rest_route(
                $this->namespace, '/' . $this->rest_base . '/documents', array(
            array(
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => array($this, 'createHubPackage')
            ),
            array(
                'methods' => WP_REST_Server::EDITABLE,
                'callback' => array($this, 'pushPackage')
            ),
            array(
                'methods' => WP_REST_Server::READABLE,
                'callback' => array($this, 'pullPackage')
            ))
        );
        
        register_rest_route(
                $this->namespace, '/' . $this->rest_base . '/documents/(?P<name>[a-z0-9.%/|*_-]+)', array(
            array(
                'methods' => WP_REST_Server::READABLE,
                'callback' => array($this, 'getFile')
            ))
        );
        
        register_rest_route(
                $this->namespace, '/' . $this->rest_base, array(
            array(
                'methods' => WP_REST_Server::READABLE,
                'callback' => array($this, 'getDraft')
            ))
        );
    }
    
    /**
     * GET: Get package draft
     * @param object $request
     * @return WP_Error|WP_REST_Response
     */
    public function getDraft($request) {
        try {
            $wpSession = BD()->api->authentication->session_key;
            BD()->api->authentication->blended_session_setter($wpSession);
            $slug = strtolower($request['slug']);
            $accountSlug = strtolower($request['accountSlug']);
            $packageJptf = $this->itinerary_route->get_getDraft($accountSlug, $slug, $wpSession);
            $response = rest_ensure_response($packageJptf);
            $response->set_status(200);
            return $response;
        } catch (Exception $e) {
            $error = json_decode($e->getMessage(), true);
            return new WP_Error($error['status_code'], $error['message'], "");
        }
    }
    
    /**
     * GET: Get package object from Hub and save it to local
     * @param object $request
     * @return WP_Error|WP_REST_Response
     */
    public function pullPackage($request) {
        try {
            $wpSession = BD()->api->authentication->session_key;
            BD()->api->authentication->blended_session_setter($wpSession);
            $slug = strtolower($request['slug']);
            $accountSlug = strtolower($request['accountSlug']);
            $force = $request['force'];
            $asHash = $request['as_hash'];
            if (isset($request['version'])) {
                $label = $request['version'];
            } else {
                $label = null;
            }
            $packageJptf = $this->itinerary_route->get_pull_package($accountSlug, $slug, $force, $asHash, $label, $wpSession);
            $response = rest_ensure_response($packageJptf);
            $response->set_status(200);
            return $response;
        } catch (Exception $e) {
            if ($e->getMessage() == "Invalid Package or package dependencies detected!") {
                return new WP_Error('500', $e->getMessage(), "");
            }
            $error = json_decode($e->getMessage(), true);
            return new WP_Error($error['status_code'], $error['message'], "");
        }
    }

    /**
     * GET: Get file jptf
     * @param object $request
     * @return WP_Error|WP_REST_Response
     */
    public function getFile($request) {
        try {
            $wpSession = BD()->api->authentication->session_key;
            BD()->api->authentication->blended_session_setter($wpSession);
            $slug = strtolower($request['slug']);
            $accountSlug = strtolower($request['accountSlug']);
            $filePath = $request['name'];
            $filePath = urldecode(urldecode($filePath));
            $filePath = ltrim(str_replace('*', '/', $filePath), '/');
            $fileJptf = $this->itinerary_route->get_getFile($accountSlug, $slug, $filePath, $wpSession);
            $response = rest_ensure_response($fileJptf);
            $response->set_status(200);
            return $response;
        } catch (Exception $e) {
            $error = json_decode($e->getMessage(), true);
            return new WP_Error($error['status_code'], $error['message'], "");
        }
    }
    
    /**
     * POST: Push package clone to Hub
     * @param object $request
     * @return WP_Error|WP_REST_Response
     */
    public function createHubPackage($request) {
        try {
            $wpSession = BD()->api->authentication->session_key;
            BD()->api->authentication->blended_session_setter($wpSession);
            $slug = strtolower($request['slug']);
            $accountSlug = strtolower($request['accountSlug']);
            $packageJptf = $this->itinerary_route->post_createHubPackage($accountSlug, $slug, $wpSession);
            $response = rest_ensure_response($packageJptf);
            $response->set_status(200);
            return $response;
        } catch (Exception $e) {
            $error = json_decode($e->getMessage(), true);
            if (!isset($error['status_code']) && !isset($error['message'])) {
                return new WP_Error("4035", "Invalid media type.", $error['data']);
            } elseif (isset($error['status_code']) && trim($error['status_code']) == "5059") {
                $this->itinerary_route->delete_localDelete($accountSlug, $slug, $wpSession);
                return new WP_Error($error['status_code'], "Package name already in use.", "");
            } elseif (isset($error['status_code']) && trim($error['status_code']) == "5064") {
                $this->itinerary_route->delete_localDelete($accountSlug, $slug, $wpSession);
                return new WP_Error($error['status_code'], $error['message'], "");
            } elseif (isset($error['status_code']) && trim($error['status_code']) == "500") {
                $this->itinerary_route->delete_localDelete($accountSlug, $slug, $wpSession);
                return new WP_Error($error['status_code'], $error['status_code'], "");
            } elseif (isset($error['message']) && trim($error['message']) == "Draft has been changed. Please pull it first") {
                $this->itinerary_route->delete_localDelete($accountSlug, $slug, $wpSession);
                return new WP_Error($error['status_code'], "Package name already in use.", "");
            } else {
                return new WP_Error($error['status_code'], $error['message'], "");
            }
        }
    }
    
    /**
     * PUT: Push package object to Hub
     * @param object $request
     * @return WP_Error|WP_REST_Response
     */
    public function pushPackage($request) {
        try {
            $wpSession = BD()->api->authentication->session_key;
            BD()->api->authentication->blended_session_setter($wpSession);
            $body = $this->get_body($request);
            $slug = strtolower($request['slug']);
            $accountSlug = strtolower($request['accountSlug']);
            $force = isset($request['force']) && !empty($request['force']) ? true : false;
            $pushToHub = $request['pushToHub'];
            $packageJptf = $this->itinerary_route->put_push_package($accountSlug, $slug, $force, $pushToHub, $body, $wpSession);
            $response = rest_ensure_response($packageJptf);
            $response->set_status(200);
            return $response;
        } catch (Exception $e) {
            try {
                $packageJptf = $this->itinerary_route->put_push_package_helper($accountSlug, $slug, $wpSession, $e->getMessage());
                return $packageJptf;
            } catch (Exception $e) {
                $error = json_decode($e->getMessage(), true);
                if (isset($error['changemessage']) && $error['changemessage'] == true && isset($error['status_code']) && isset($error['message'])) {
                    $error['message'] = "You need to sync.";
                    return new WP_Error($error['status_code'], $error['message'], "");
                }
                if (isset($error['changemessage']) && $error['changemessage'] == true) {
                    return new WP_Error('500', "Invalid Package. Something wrong with your package either the file _package.json not found or you are trying to push some invalid files.", "");
                }
                if (isset($error['code']) && trim($error['code']) == "4035") {
                    return new WP_Error($error['code'], $error['message'], $error['data']);
                }
                if (isset($error['data']) && !isset($error['status_code']) && !isset($error['message'])) {
                    $dataValue = "Upload a valid image. The file you uploaded was either not an image or a corrupted image.";
                    if (is_array($error['data']) && !empty($error['data']))
                        $error['data'] = array_fill_keys(array_keys($error['data']), $dataValue);
                    return new WP_Error("4035", "Invalid media type.", $error['data']);
                }
                if (isset($error['errors'])) {
                    return new WP_Error($error['status_code'], $error['errors'], "");
                } else if (isset($error['status_code']) && isset($error['message']) && isset($error['data'])) {
                    return new WP_Error($error['status_code'], $error['message'], $error['data']);
                } else if (isset($error['status_code']) && isset($error['message'])) {
                    return new WP_Error($error['status_code'], $error['message'], "");
                } else {
                    return new WP_Error('500', 'Package Does not Exist.', "");
                }
            }
        }
    }

}
