<?php

/**
 * Blended REST Packages Versions Controller 
 *
 * @class    Blended
 * @author   Blended
 * @package  Blended/Rest/Packages
 * @version  1.0.0
 */
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

class BD_REST_Packages_Versions_Controller extends BD_REST_Packages_Controller {

    /**
     * Endpoint
     * @var string
     */
    protected $rest_base = 'packages/(?P<accountSlug>[a-z0-9_\-]+)/(?P<slug>[a-z0-9_\-]+)/versions';

    public function register_routes() {
        register_rest_route(
                $this->namespace, '/' . $this->rest_base, array(
            array(
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => array($this, 'versionCreateSnapshot')
            ),
            array(
                'methods' => WP_REST_Server::READABLE,
                'callback' => array($this, 'getVersions')
            ),)
        );
        
        register_rest_route(
                $this->namespace, '/' . $this->rest_base . '/(?P<label>[a-z0-9._\-]+)/licenses/(?P<name>[a-z0-9.%/|*_-]+)', array(
            array(
                'methods' => WP_REST_Server::EDITABLE,
                'callback' => array($this, 'updateLicense')
            ),
            array(
                'methods' => WP_REST_Server::DELETABLE,
                'callback' => array($this, 'removeLicense')
            ),
            )
        );
        
         register_rest_route(
                $this->namespace, '/' . $this->rest_base . '/(?P<label>[a-z0-9._\-]+)', array(
            array(
                'methods' => WP_REST_Server::READABLE,
                'callback' => array($this, 'getVersionDetails')
            ))
        );
         
         register_rest_route(
                $this->namespace, '/' . $this->rest_base . '/(?P<label>[a-z0-9._\-]+)/documents', array(
            array(
                'methods' => WP_REST_Server::READABLE,
                'callback' => array($this, 'versionDownload')
            ))
        );
         
         register_rest_route(
                $this->namespace, '/' . $this->rest_base . '/(?P<label>[a-z0-9._\-]+)/documents/(?P<name>[a-z0-9.%/|*_-]+)', array(
            array(
                'methods' => WP_REST_Server::READABLE,
                'callback' => array($this, 'versionDownloadFile')
            ))
        );
         
         register_rest_route(
                $this->namespace, '/' . $this->rest_base . '/(?P<label>[a-z0-9._\-]+)/licenses', array(
            array(
                'methods' => WP_REST_Server::READABLE,
                'callback' => array($this, 'getLicenses')
            ))
        );
        
        
        
        
    }
    
    /**
     * POST: Create snapshot of a package
     * @param object $request
     * @return WP_Error|WP_REST_Response
     */
    public function versionCreateSnapshot($request) {
        try {
            $wpSession = BD()->api->authentication->session_key;
            BD()->api->authentication->blended_session_setter($wpSession);
            $body = $this->get_body($request);
            $slug = strtolower($request['slug']);
            $accountSlug = strtolower($request['accountSlug']);
            $packageDetails = $this->itinerary_route->post_versionCreateSnapshot($accountSlug, $slug, $body, $wpSession);
            $response = rest_ensure_response($packageDetails);
            $response->set_status(200);
            return $response;
        } catch (Exception $e) {
            $code = $e->getCode();
            $error = json_decode($e->getMessage(), true);
            $message = (isset($error['message']) ? $error['message'] : "");
            $status_code = (isset($error['status_code']) ? $error['status_code'] : "");
            $draft_dependency = isset($error['draft_dependency']) && !empty($error['draft_dependency']) ? array("draft_dependency" => $error['draft_dependency']) : "";
            if (!empty($draft_dependency)) {
                return new WP_Error($error['status_code'], $error['message'], $draft_dependency);
            } else if ($code === 500) {
                return new WP_Error($error['status_code'], $error['message'], "");
            }
            $data = prepare_response_data($error, $code);
            return new WP_Error($status_code, $message, $data);
        }
    }

    /**
     * GET: Retrieve list of package version
     * @param object $request
     * @return WP_Error|WP_REST_Response
     */
    public function getVersions($request) {
        try {
            $wpSession = BD()->api->authentication->session_key;
            BD()->api->authentication->blended_session_setter($wpSession);
            $accountSlug = strtolower($request['accountSlug']);
            $slug = strtolower($request['slug']);
            $versionList = $this->itinerary_route->get_getVersions($accountSlug, $slug, $wpSession);
            $response = rest_ensure_response($versionList);
            $response->set_status(200);
            return $response;
        } catch (Exception $e) {
            $error = json_decode($e->getMessage(), true);
            return new WP_Error($error['status_code'], $error['message'], "");
        }
    }
    
    /**
     * GET: Get version data of a package in details
     * @param object $request
     * @return WP_Error|WP_REST_Response
     */
    public function getVersionDetails($request) {
        try {
            $wpSession = BD()->api->authentication->session_key;
            BD()->api->authentication->blended_session_setter($wpSession);
            $slug = strtolower($request['slug']);
            $accountSlug = strtolower($request['accountSlug']);
            $label = $request['label'];
            $packageDetails = $this->itinerary_route->get_getVersionDetails($accountSlug, $slug, $label, $wpSession);
            $response = rest_ensure_response($packageDetails);
            $response->set_status(200);
            return $response;
        } catch (Exception $e) {
            $error = json_decode($e->getMessage(), true);
            return new WP_Error($error['status_code'], $error['message'], "");
        }
    }
    
    /**
     * GET: Get version documents of a package
     * @param object $request
     * @return WP_Error|WP_REST_Response
     */
    public function versionDownload($request) {
        try {
            $wpSession = BD()->api->authentication->session_key;
            BD()->api->authentication->blended_session_setter($wpSession);
            $slug = strtolower($request['slug']);
            $accountSlug = strtolower($request['accountSlug']);
            $label = $request['label'];
            $asHash = $request['as_hash'];
            $packageJptf = $this->itinerary_route->get_versionDownload($accountSlug, $slug, $label, $asHash, $wpSession);
            $response = rest_ensure_response($packageJptf);
            $response->set_status(200);
            return $response;
        } catch (Exception $e) {
            $error = json_decode($e->getMessage(), true);
            return new WP_Error($error['status_code'], $error['message'], "");
        }
    }
    
    /**
     * GET: Get version documents of a package in details
     * @param object $request
     * @return WP_Error|WP_REST_Response
     */
    public function versionDownloadFile($request) {
        try {
            $wpSession = BD()->api->authentication->session_key;
            BD()->api->authentication->blended_session_setter($wpSession);
            $slug = strtolower($request['slug']);
            $accountSlug = strtolower($request['accountSlug']);
            $label = $request['label'];
            $filePath = $request['name'];
            $filePath = urldecode(urldecode($filePath));
            $filePath = ltrim(str_replace('*', '/', $filePath), '/');
            $packageJptf = $this->itinerary_route->get_versionDownloadFile($accountSlug, $slug, $label, $filePath, $wpSession);
            $response = rest_ensure_response($packageJptf);
            $response->set_status(200);
            return $response;
        } catch (Exception $e) {
            $error = json_decode($e->getMessage(), true);
            return new WP_Error($error['status_code'], $error['message'], "");
        }
    }
    
    /**
     * GET: Get list of licenses associated with package
     * @param object $request
     * @return WP_Error|WP_REST_Response
     */
    public function getLicenses($request) {
        try {
            $wpSession = BD()->api->authentication->session_key;
            BD()->api->authentication->blended_session_setter($wpSession);
            $accountSlug = strtolower($request['accountSlug']);
            $slug = strtolower($request['slug']);
            $label = $request['label'];
            $licenseDetails = $this->itinerary_route->get_getLicenses($accountSlug, $slug, $label, $wpSession);
            $response = rest_ensure_response($licenseDetails);
            $response->set_status(200);
            return $response;
        } catch (Exception $e) {
            $error = json_decode($e->getMessage(), true);
            return new WP_Error($error['status_code'], $error['message'], "");
        }
    }

    /**
     * PUT: Update package license
     * @param object $request
     * @return WP_Error|WP_REST_Response
     */
    public function updateLicense($request) {
        try {
            $wpSession = BD()->api->authentication->session_key;
            BD()->api->authentication->blended_session_setter($wpSession);
            $body = $this->get_body($request);
            $accountSlug = strtolower($request['accountSlug']);
            $slug = strtolower($request['slug']);
            $label = $request['label'];
            $name = $request['name'];
            $licenseDetails = $this->itinerary_route->put_updateLicense($accountSlug, $slug, $label, $name, $body, $wpSession);
            $response = rest_ensure_response($licenseDetails);
            $response->set_status(200);
            return $response;
        } catch (Exception $e) {
            $error = json_decode($e->getMessage(), true);
            return new WP_Error($error['status_code'], $error['message'], "");
        }
    }

    
    
    /**
     * DELETE: Reject shared package
     * @param object $request
     * @return WP_Error|WP_REST_Response
     */
    public function removeLicense($request) {
        try {
            $wpSession = BD()->api->authentication->session_key;
            BD()->api->authentication->blended_session_setter($wpSession);
            $accountSlug = strtolower($request['accountSlug']);
            $slug = strtolower($request['slug']);
            $label = $request['label'];
            $name = $request['name'];
            $license = $this->itinerary_route->delete_removeLicense($accountSlug, $slug, $label, $name, $wpSession);
            $response = rest_ensure_response($license);
            $response->set_status(200);
            return $response;
        } catch (Exception $e) {
            $error = json_decode($e->getMessage(), true);
            return new WP_Error($error['status_code'], $error['message'], "");
        }
    }

}
