<?php

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

use Blended\hostlib\TransformImage;

/* Blended Helper Functions
 *  @since 1.0
 */

/* Returns the site URL of the home pagenow
 *  @param none
 *  @returns string containing the home page URL
 */

function blended_home_url() {
    return network_site_url();
}

/**
 * Theme activation tasks
 * @param string $code
 * @param string $function
 */
function wp_register_theme_activation_hook($code, $function) {
    $optionKey = "theme_is_activated_" . $code;
    if (!get_option($optionKey)) {
        call_user_func($function);
        update_option($optionKey, 1);
    }
}

/**
 * Get menu page url
 * @todo add it to Rest functions
 * @param string $endPointName
 * @return string
 * @throws \Exception
 */
function blendedAppUrl($endPointName) {
    try {
        if (!$endPointName)
            throw new \Exception(json_encode(array("code" => 500, "message" => 'Please provide endpoint name')));

        $url = "";
        switch ($endPointName) {

            case "Library":
                $url = admin_url('admin.php?page=' . "Library");
                break;
            case "SourceCode":
                $url = admin_url('admin.php?page=' . "Source+Code");
                break;
            case "Onboard":
                $url = admin_url('admin.php?page=' . "Onboard");

                break;
            case "Settings":
                $url = admin_url('admin.php?page=' . "Setting");
                break;
            case "Designer":
                $url = admin_url('admin.php?page=' . "designer");
                break;
        }
    } catch (\Exception $e) {
        throw $e;
    }

    $url = json_encode($url, JSON_UNESCAPED_SLASHES);
    $url = json_decode($url, true);
    return $url;
}

/**
 * Get rest prefix
 * @global object $blendedConfig
 * @return string Rest Prefix
 */
function blended_api() {
    global $blendedConfig;
    $prefix = $blendedConfig->get('swagger', 'BASE_PATH_1');
    return $prefix;
}

/**
 * Get rest namespace
 * @global object $blendedConfig
 * @return string Rest namespace
 */
function blended_rest_namespace() {
    global $blendedConfig;
    return $blendedConfig->get('swagger', 'BASE_PATH_2');
}

/**
 * Create media temporary file
 * @param array $args
 * @return boolean
 */
function create_temp_file(Array $args) {
    if (empty($args)) {
        return false;
    }
    $filename = $args["hash"] . "_" . $args["height"] . "_" . $args["width"];
    $filename = ($args["filters"] ? $filename . "_" . $args["filters"] : $filename);
    $filename = $filename . ".tmp.txt";
    $filepath = BD()->config['media_meta_path'] . "/" . $filename;
    if (file_exists($filepath)) {
        return false;
    }
    validate_dir_limit(BD()->config['media_meta_path'], BD()->config['media_meta_limt']);
    $args["created_at"] = time();
    if (isset($args['hash']))
        unset($args['hash']);
    $data = json_encode($args);
    $data = BD()->backend->_indent($data);
    return file_put_contents($filepath, $data);
}

function get_cache_path(Array $args) {
    $file = get_cache_file($args);
    if ($file) {
        return BD()->config['cache_path'] . "/" . $file;
    }
    return false;
}

function get_cache_uri(Array $args) {
    $file = get_cache_file($args);
    if ($file) {
        return BD()->config['cache_uri'] . "/" . $file;
    }
    return false;
}

function get_cache_file(Array $args) {
    if (empty($args)) {
        return false;
    }
    $file_extension = BD()->backend()->getLastString($args['filename']);
    $filename = $args["hash"] . "_" . $args["height"] . "x" . $args["width"];
    $filename = ($args["filters"] ? $filename . "x" . $args["filters"] : $filename);
    $filename = $filename . "." . $file_extension;
    return $filename;
}

function encrypt_decrypt($action, $string) {
    $output = false;

    $encrypt_method = "AES-256-CBC";
    $secret_key = SECURE_AUTH_KEY;
    $secret_iv = AUTH_KEY;

    // hash
    $key = hash('sha256', $secret_key);

    // iv - encrypt method AES-256-CBC expects 16 bytes - else you will get a warning
    $iv = substr(hash('sha256', $secret_iv), 0, 16);

    if ($action == 'encrypt') {
        $output = openssl_encrypt($string, $encrypt_method, $key, 0, $iv);
        $output = base64_encode($output);
    } else if ($action == 'decrypt') {
        $output = openssl_decrypt(base64_decode($string), $encrypt_method, $key, 0, $iv);
    }

    return $output;
}

/**
 * Validate the file count inside a directory
 * LRU Based
 * @param string $directory
 * @param integer $limit 
 * @return none
 */
function validate_dir_limit($directory, $limit) {
    if (!is_dir($directory) || !absint($limit)) {
        return;
    }
    $files = array_diff(scandir($directory), array('.', '..'));
    if (count($files) > $limit) {
        array_multisort(
                array_map('filemtime', $files),
                SORT_NUMERIC,
                SORT_ASC,
                $files
        );
        unlink($directory . "/" . $files[0]);
    }
}

/* blended image function that can call Image_Transform class
  to make transformations on images and store it into cache directory. */

/**
 * Get image src
 * @global string $image_specif_path
 * @param string $image_url
 * @param string $filters
 * @param int $height
 * @param int $width
 * @return boolean | string
 */
function image($image_data, $height = 0, $width = 0, $filters = null) {
    // Image url/path
    $image_url = isset($image_data['data']) ? $image_data['data'] : "";

    if (!$image_url) {
        return false;
    }

    $image_filters = compact("height", "width", "filters");
    /**
     * Check if cache file exists 
     * @return url cached file
     */
    $cached_path = get_cache_path(array_merge($image_data, $image_filters));
    $cached_uri = get_cache_uri(array_merge($image_data, $image_filters));
    if (file_exists($cached_path)) {
        return $cached_uri;
    }

    //return image endpoint if $image_url is a url
    if (filter_var($image_url, FILTER_VALIDATE_URL)) {
        create_temp_file(array_merge([
            "ref_part" => $image_data['ref_part'],
            "hash" => $image_data['hash']
                        ], $image_filters));
        $params = $height . "_" . $width;
        $params = ($filters ? $params . "_" . $filters : $params);
        $params = encrypt_decrypt("encrypt", $params);
        return rest_url(blended_rest_namespace() . '/' . sprintf('packages/local/media/%s/%s', $image_data['hash'], $params));
    }

    /* Applying effects to image, if does not found any image in the cache directory. */
    if (!file_exists($cached_path)) {
        validate_dir_limit(BD()->config['cache_path'], BD()->config['cache_file_limit']);
        $obj = new TransformImage($image_url, $width, $height);

        if (!empty($width) || !empty($height)) {
            $filters = 'series(fill, crop(smart))';
        } else {
            $filters = null;
        }

        // applying filters here.
        if (!empty($filters)) {
            if (strpos($filters, "series") != false) {
                $img_obj = $obj->apply_filter($filters);
            } else {
                $filters = str_replace('series(', '', $filters);
                $filters = rtrim($filters, ')');
                $img_obj = $obj->series($filters);
            }
        } else {
            $img_obj = $obj->img;
        }
        // saving image object to cache directory.
        switch ($image_data["ext"]) {
            case "jpg":
                imagejpeg($img_obj, $cached_path, 100);
                break;
            case "jpeg":
                imagejpeg($img_obj, $cached_path, 100);
                break;
            case "bmp":
                imagewbmp($img_obj, $cached_path);
                break;
            case "png":
                imagesavealpha($img_obj, true);
                imagepng($img_obj, $cached_path, 0);
                break;
            case "gif":
                imagegif($img_obj, $cached_path);
                break;
            case "webp":
                imagewebp($img_obj, $cached_path);
                break;
            case "default":
                imagejpeg($img_obj, $cached_path, '100');
                break;
        }
        // discarding image object after saving it to cache directory.
        if (isset($img_obj)) {
            imagedestroy($img_obj);
        }
    }
    
    if ($cached_uri) {
        return $cached_uri;
    }
    return false;
}

function css_links($theme) {
    $context['theme'] = $theme;

    try {
        if (isset($theme['meta']['css']['compile_targets']['0'])) {
            $compiled_css = Timber::compile_string($theme['css']['blendedcss'], $context);
            $temp_style = fopen("blended_style.css", "w");
            fwrite($temp_style, $compiled_css);
            fclose($temp_style);
            $css_path = site_url() . "/blended_style.css";
            $style = $css_path;
            return $style;
        }
    } catch (\Exception $e) {
        $msg = "Exception " . $e->getCode() . " / " . $e->getMessage();
        echo "<p>$msg</p>";
    }
}

function filter_writable_props($schema) {
    return empty($schema['readonly']);
}

function prepare_response_data($error, $code) {
    switch ($code) {
        case 409:
            $schema = error_response_409_schema();
            break;
        case 412:
            $schema = error_response_412_schema();
            break;
        case 401:
            $schema = error_response_401_schema();
            break;
        case 400:
            $schema = error_response_400_schema();
            break;
        case 500:
            // $data = error_response_500_schema();
            return [];
        default:
            break;
    }

    $data = [];
    $data_keys = array_keys(array_filter($schema['properties'], 'filter_writable_props'));
    foreach ($data_keys as $key) {
        $data[$key] = $error[$key];
    }
    return $data;
}

function error_response_409_schema() {
    $schema = array(
        '$schema' => 'http://json-schema.org/draft-04/schema#',
        'title' => "Rest response for HTTP-409",
        'type' => 'object',
        'properties' => array(
            'errors' => array(
                'description' => __('Validation error.', 'blended'),
                'type' => 'array',
                'items' => array('type' => 'string',),
            ),
            'status_code' => array(
                'description' => __('Error custom status code.', 'blended'),
                'type' => 'integer',
                'format' => 'int32',
                'readonly' => true
            ),
            'purchase_packages_required' => array(
                'description' => __("Validation error.", 'blended'),
                'type' => 'array',
                'items' => array(
                    'type' => 'string',
                ),
            ),
            'purchase_packages_given' => array(
                'description' => __("Validation error.", 'blended'),
                'type' => 'array',
                'items' => array(
                    'type' => 'string',
                ),
            ),
            'auto_share_packages_required' => array(
                'description' => __("Validation error.", 'blended'),
                'type' => 'array',
                'items' => array(
                    'type' => 'string',
                ),
            ),
            'auto_share_packages_given' => array(
                'description' => __("Validation error.", 'blended'),
                'type' => 'array',
                'items' => array(
                    'type' => 'string',
                ),
            ),
        ),
    );
    return $schema;
}

function error_response_412_schema() {
    $schema = array(
        '$schema' => 'http://json-schema.org/draft-04/schema#',
        'title' => "Rest response for HTTP-412",
        'type' => 'object',
        'properties' => array(
            'error' => array(
                'description' => __('Validation error.', 'blended'),
                'type' => 'array',
                'items' => array('type' => 'string',),
            ),
            'status_code' => array(
                'description' => __('Error custom status code.', 'blended'),
                'type' => 'integer',
                'format' => 'int32',
                'readonly' => true
            ),
            'message' => array(
                'description' => __('Error message.', 'blended'),
                'type' => 'string',
                'readonly' => true
            ),
            'items' => array(
                'description' => __("Validation error. List of private dependencies.", 'blended'),
                'type' => 'array',
                'items' => array(
                    'type' => 'string',
                ),
            ),
            'thumbnail_image_data' => array(),
            'data' => array(
                'description' => __("Validation error. List of paid dependencies.", 'blended'),
                'type' => 'array',
                'items' => array(
                    'type' => 'string',
                ),
            ),
            'redirect_url' => array(
                'type' => 'string',
                'description' => __("URL if redirection to website is needed at all.", 'blended'),
            ),
            'validator_sets_to_pass' => array(
                'type' => 'array',
                'description' => __("Need to pass at least one complete set of validators.", 'blended'),
                'items' => array(
                    'type' => 'string',
                ),
            ),
        ),
    );
    return $schema;
}

function error_response_401_schema() {
    $schema = array(
        '$schema' => 'http://json-schema.org/draft-04/schema#',
        'title' => "Rest response for HTTP-401",
        'type' => 'object',
        'properties' => array(
            'error' => array(
                'description' => __('Validation error.', 'blended'),
                'type' => 'array',
                'items' => array('type' => 'string',),
            ),
            'status_code' => array(
                'description' => __('Error custom status code.', 'blended'),
                'type' => 'integer',
                'format' => 'int32',
                'readonly' => true
            ),
        ),
    );
    return $schema;
}

function error_response_400_schema() {
    $schema = array(
        '$schema' => 'http://json-schema.org/draft-04/schema#',
        'title' => "Rest response for HTTP-400",
        'type' => 'object',
        'properties' => array(
            'errors' => array(
                'description' => __('Validation error.', 'blended'),
                'type' => 'object',
            ),
            'status_code' => array(
                'description' => __('Error custom status code.', 'blended'),
                'type' => 'integer',
                'format' => 'int32',
                'readonly' => true
            ),
        ),
    );
    return $schema;
}

function error_response_500_schema() {
    $schema = array(
        '$schema' => 'http://json-schema.org/draft-04/schema#',
        'title' => "Rest response for HTTP-500",
        'type' => 'object',
        'properties' => array(
            'status_code' => array(
                'description' => __('Error custom status code.', 'blended'),
                'type' => 'integer',
                'format' => 'int32',
                'readonly' => true
            ),
            'message' => array(
                'description' => __('Error message.', 'blended'),
                'type' => 'string',
                'readonly' => true
            ),
        ),
    );
    return $schema;
}
