<?php
/**
 * Blended Theme Framework Class
 *
 * @class    Blended
 * @author   Blended
 * @package  blended_fw/inc
 * @version  1.0.0
 */
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

if (!class_exists('Blended_FW')) :

    final class Blended_FW {

        /**
         * Theme version.
         *
         * @var string
         */
        public $version = '1.0.0';

        /**
         * The single instance of the class.
         */
        protected static $_instance = null;

        /**
         * Backend instance
         * @var Backend Object 
         */
        public $backend = null;

        /**
         * Controller instance
         * @var Controller Object 
         */
        public $controller = null;

        /**
         * Network instance
         * @var Network Object 
         */
        public $network = null;

        /**
         * Blended Loader
         * @var Blended_Loader object 
         */
        public $loader = null;

        /**
         * Backend Class Constructor Argument
         * @var string
         */
        public $backend_slug = null;

        /**
         * Network key
         * @var string
         */
        public $network_key = null;

        /**
         * Blended Theme Framework Instance.
         *
         * @return Main instance.
         */
        public static function instance() {
            if (is_null(self::$_instance)) {
                self::$_instance = new self();
            }
            return self::$_instance;
        }

        /**
         * Constructor.
         */
        public function __construct() {

            //Check if required plugins are loaded
            if (!$this->plugins_installed())
                return;
            $this->define_config();
            $this->includes();
            $this->init_hooks();


            do_action('blended_loaded');
        }

        /**
         * Hooks.
         */
        private function init_hooks() {
            add_action('init', array($this, '_ini_set'));
            add_action('after_setup_theme', array($this, 'setup'));
            add_action('wp_logout', array($this->loader, 'clearLocalStorage'));
            add_action('widgets_init', array($this, 'widgets_init'));
            add_action("wp_enqueue_scripts", array($this, "scripts"));
            add_filter('site_transient_update_plugins', array($this, 'plugin_updates'));
        }

        /**
         * Include required files.
         */
        public function includes() {
            require_once BD_PATH . '/inc/class-autoloader.php';
            require_once BD_PATH . '/inc/class-blended-post-types.php';

            //Load Blended Framework
            $this->loader = require_once BD_PATH . '/inc/class-bd-loader.php';


            //Include admin instances.
            if (is_admin())
                require_once BD_PATH . '/inc/admin/class-blended-admin.php';

            require_once BD_PATH . '/inc/article_functions.php';
            require_once BD_PATH . '/inc/navigation_functions.php';
            require_once BD_PATH . '/inc/class-blended-shortcodes.php';
            require_once BD_PATH . '/inc/class-wc-support.php';
            require_once BD_PATH . '/inc/comment-functions.php';

            //REST API
            require_once BD_PATH . '/inc/class-bd-api.php';
            $this->api = new BD_API();
        }

        private function define_config() {
            $this->config = require_once BD_PATH . '/inc/config.php';
        }
       
        public function setup() {

            /**
             * Posts and comments RSS feed links to head.
             */
            add_theme_support('automatic-feed-links');

            /*
             * Post Thumbnails on posts and pages.
             */
            add_theme_support('post-thumbnails');

            //Nav Menus.
            register_nav_menus(array(
                'primary' => __('Primary Menu', 'blended'),
                'secondary' => __('Secondary Menu', 'blended')
            ));

            //Custom Image Sizes
            add_image_size('admin-list-thumb', 80, 80, true);
            add_image_size('gallery-list-thumb', 360, 260, true);
        }

        public function widgets_init() {

            register_sidebar(array(
                'name' => __('Main Widget Area', 'blended'),
                'id' => 'sidebar-1',
                'description' => __('Appears in the footer section of the site.', 'blended'),
                'before_widget' => '<aside id="%1$s" class="widget %2$s">',
                'after_widget' => '</aside>',
                'before_title' => '<h3 class="widget-title">',
                'after_title' => '</h3>'
            ));

            register_sidebar(array(
                'name' => __('Secondary Widget Area', 'blended'),
                'id' => 'sidebar-2',
                'description' => __('Appears on posts and pages in the sidebar.', 'blended'),
                'before_widget' => '<aside id="%1$s" class="widget %2$s">',
                'after_widget' => '</aside>',
                'before_title' => '<h3 class="widget-title">',
                'after_title' => '</h3>'
            ));

            register_sidebar(array(
                'name' => __('Footer Widget Area', 'blended'),
                'id' => 'sidebar-3',
                'description' => __('Appears on Footer Area.', 'blended'),
                'before_widget' => '<div class="one_third-md footer-content__col">',
                'after_widget' => '</div>',
                'before_title' => '<h3 class="widget-title">',
                'after_title' => '</h3>'
            ));

            register_sidebar(array(
                'name' => __('Contact Map', 'blended'),
                'id' => 'sidebar-4',
                'description' => __('Appears on Contact Page.', 'blended'),
                'before_widget' => '<div class="map-responsive">',
                'after_widget' => '</div>',
                'before_title' => '<h3 class="widget-title">',
                'after_title' => '</h3>'
            ));
        }

        /**
         * Load Scripts
         */
        public function scripts() {
            wp_enqueue_script('jquery');
        }

        public function plugins_installed() {

            //First install required plugins
            require_once BD_PATH . '/inc/tgmaBlended.php';

            /**
             * Backward compatibility
             * Now check if any plugin is deactive
             */
            $required_plugins = array(
                array(
                    "key" => "timber-library/timber.php",
                    "notice" => __(sprintf("Blended themes requires %s plugin to be activated ", "<strong>Timber</strong>"), "blended"),
                    "class" => "TimberSite"
                )
            );

            $all_set = true;
            foreach ($required_plugins as $plugin):

                /**
                 * @todo Check network installation if required by using 
                 * || is_plugin_active_for_network( $plugin['key'] ) in if
                 */
                if (!in_array($plugin['key'], (array) apply_filters('active_plugins', get_option('active_plugins'))) || !class_exists($plugin['class'])):
                    $notice = $plugin['notice'];

                    //Admin notice
                    //add_action('admin_notices', function() use($notice) {
                    //    printf('<div class="%1$s"><p>%2$s</p></div>', esc_attr("notice notice-error"), $notice);
                    //});
                    //Frontend notice
                    if (!is_admin())
                        wp_die($notice);

                    //Looks like this plugin is not activated
                    $all_set = false;
                endif;

            endforeach;

            return $all_set;
        }

        /**
         * Manage plugin updates
         * Hides update nag
         * @param object $value
         * @return object
         */
        public function plugin_updates($value) {
            if(isset($value->response['timber-library/timber.php']))
                unset($value->response['timber-library/timber.php']);
            return $value;
        }

        /**
         * Set precision and serialize_precision
         * for php verion greater than eqal to
         * 7.1
         */
        public function _ini_set() {
            if (version_compare(phpversion(), '7.1', '>=')) {
               ini_set( 'precision', 14 );
               ini_set( 'serialize_precision', -1 );
            }
        }
        
        /**
         * Get Backend Class
         * @param string $slug
         * @return Backend Object
         */
        public function backend($slug = null) {
            global $blendedConfig;
            $switch = $blendedConfig->get("setup_configuration", "BACKEND");
            $this->backend_slug = $slug;
            switch ($switch) {
                case "DB":
                    $backend = new Blended\Orm\DB_Backend($this->backend_slug);
                    return $this->backend = $backend;
                case "FS":
                    $backend = new Blended\hostlib\Backend($this->backend_slug);
                    return $this->backend = $backend;
                default:
                    $backend = new Blended\hostlib\Backend($this->backend_slug);
                    return $this->backend = $backend;
            }
        }

        /**
         * Get Network class 
         * @param string $sessionKey
         * @return Network Object
         */
        public function network($sessionKey = null) {
            $this->network = new Blended\hostlib\Network($sessionKey);
            $this->network_key = $this->network->sessionKey;
            return $this->network;
        }

        /**
         * Get Controller Class
         * @return Controller Object
         */
        public function controller() {
            if (!$this->backend_slug)
                $this->backend();

            if (!$this->network_key)
                $this->network();

            return $this->controller = new Blended\hostlib\Controller($this->backend, $this->network);
        }

    }

    endif;

function BD() {
    return Blended_FW::instance();
}
