<?php
/**
 * Handles Blended Filesystem , manage users and owner of blended themes.
 *
 * Also manages Wordpress and blended theme settings.
 *
 * @blended
 */

# Manage blended theme configuration settings.
include("Blended_ini_Parser.php");

# All path setting defined here.
$config = new Blended_ini_Parser("config.ini");

# React App theme Settings are stored here.
$theme_settings = new Blended_ini_Parser("theme_settings.ini");

# Path to theme directory.
$THEME_DIRECTORY = dirname(__FILE__) . '/..';

# Current active theme.
$ACTIVE_THEME = $config->get("setup_configuration", "ACTIVE_THEME");

# Anonymous user works without login.
$DEFAULT_USER = $config->get("setup_configuration", "DEFAULT_USER");

# Theme directory where all themes files reside.
$ROOT_DIRECTORY = $config->get("setup_configuration", "ROOT_DIRECTORY");

# Path to the theme that is to be loaded into memory.
$ACTIVE_ROOT_DIRECTORY = $ROOT_DIRECTORY . '/' . $ACTIVE_THEME;

# Path used to save dependencies locally.
$DEPENDENCY_PATH = $config->get("dependency_loader", "DEPENDENCY_PATH");

# Path used to save dependencies locally.
$SOURCE_DIRECTORY = $config->get("setup_configuration", "SOURCE_DIRECTORY");

# Path to css directory.
$CSS_DIRECTORY = $config->get("backend_multimedia_directory", "CSS_DIRECTORY");

# Path to image cachce directory.
$CACHE_IMAGE_DIRECTORY = $THEME_DIRECTORY . '/' . $config->get("backend_multimedia_directory", "IMAGE_DIRECTORY");

# Path to javascript directory.
$JS_DIRECTORY = $config->get("backend_multimedia_directory", "JS_DIRECTORY");

# Following extension will be treated as media type.
$allowed_image_types = array(
    'jpg',
    'jpeg',
    'bmp',
    'gif',
    'tif',
    'tiff',
    'png',
    'ico',
    'webp',
    'otf',
    'eot',
    'ttf',
    'woff',
    'woff2'
);

# File system Backend Class.
class Backend
{
    
    var $ROOT_DIRECTORY;
    var $DIRECTORY_NODE;
    function __construct($ROOT_DIRECTORY = null)
    {
        $this->ROOT_DIRECTORY = $ROOT_DIRECTORY;
        $this->DIRECTORY_NODE = new DirectoryNode($this->ROOT_DIRECTORY);
    }
    
    # Load project object from local file system.
    function get_package()
    {
        $intermediary_object = $this->DIRECTORY_NODE;
        return $intermediary_object;
    }
    
    function compare_package($hub_package, $local_package, $action = null)
    {
        $project               = $this->DIRECTORY_NODE;      
        $local_last_packageHash = $project->getHash($this->ROOT_DIRECTORY);
        $hub_current_packageHash = $hub_package[0]['/']['hash'];     
        
        if ($action == "push") {
            if ($hub_current_packageHash === $local_last_packageHash) {
                return true;
            } else {
                return $this->differences($hub_package, $local_package);
            }
        } elseif ($action == "pull") {
            if ($hub_current_packageHash === $local_last_packageHash) {
                return true;
            } else {
                return $this->differences($hub_package, $local_package);
            }
        }
    }
    
    function differences($hub_package, $local_package)
    {
        $local_package_jptf = $this->as_jptf($local_package);
        $hub_package = $hub_package[0]['/']['content'];
        $local_package_jptf = $local_package_jptf[0]['/']['content'];

        return $this->blendedRecursiveDiff($hub_package, $local_package_jptf);
    }

function blendedRecursiveDiff($hub_package, $local_package, $track = null) { 
     
    $list = array();

    $new_file = array_keys($hub_package);
    if(empty($new_file)) {
       $fileAdded = array_keys($local_package[0]);
       $track = $track . '/' . $fileAdded[0];
       array_push($list, $track);              
    }

    foreach ($hub_package as $key => $value) { 
        if(is_int($key)) {
             if(!isset($local_package[$key])) {
               $deletedFile = array_keys($value);
               $track = $track . '/' . $deletedFile[0];
               array_push($list, $track);                
             } else if(!isset($hub_package[$key]) || empty($hub_package[$key])) {
                  $deletedFile = array_keys($value);
                  $track = $track . '/' . $deletedFile[0];
                  array_push($list, $track);        
             } else {
                $list = array_merge($this->blendedRecursiveDiff($hub_package[$key], $local_package[$key], $track), $list);
             }
        } elseif((strpos($key, '.') !== false)) {

            if($hub_package[$key]['hash'] != $local_package[$key]['hash']) {
               $track = $track . '/' . $key;
               array_push($list, $track);
            }            
        } elseif((strpos($key, '.') !== true)) {
            if($hub_package[$key]['hash'] != $local_package[$key]['hash']) {
               $track = $track . '/' . $key;
               $list = array_merge($this->blendedRecursiveDiff($hub_package[$key]['content'], $local_package[$key]['content'], $track), $list);
            }
        }
    } 

    return $list;
} 
    
    # Converting jptf project object to Intermediary format.
    function deserialize_jptf($jptf_project)
    {
        $packageHash  = $jptf_project[0]['/']['hash'];
        $jptf_project = $jptf_project[0]['/']['content'];
        $Intermediary = $this->blended_deserialize_jptf($jptf_project);
        $Intermediary = new DirectoryNode('', $Intermediary);
        $Intermediary->hash = $packageHash;
        $Intermediary->location = "/";
        return $Intermediary;
    }
    
    function blended_deserialize_jptf($jptf_project, $Intermediate_project = array())
    {
        global $allowed_image_types;
        try {
            foreach ($jptf_project as $key => $value) {

                $file_extension = explode('.', $key);
                $file_extension = $file_extension[count($file_extension) - 1];

                if (is_int($key)) {
                    $Intermediate_project = array_merge($Intermediate_project, $this->blended_deserialize_jptf($value));
                } elseif ((strpos($key, '.json') !== false)) {
                    $Intermediate_project[$key]           = new JsonFile(json_decode(gzuncompress(base64_decode($value['content']))));
                    $Intermediate_project[$key]->location = $key;
                } elseif (in_array(strtolower($file_extension), $allowed_image_types)) {
                    $Intermediate_project[$key]           = new BinaryFile($key);
                    $Intermediate_project[$key]->location = $key;
                    $Intermediate_project[$key]->hash = $value['hash'];
                } elseif ((strpos($key, '.') !== false)) {
                    $Intermediate_project[$key]           = new TextFile(gzuncompress(base64_decode($value['content'])));
                    $Intermediate_project[$key]->location = $key;
                } elseif ((strpos($key, '.') !== true)) {
                    $data                       = new DirectoryNode($key, $value['content']);
                    $data->hash                 = $value['hash'];
                    $data->data                 = $this->blended_deserialize_jptf($data->data);
                    $Intermediate_project[$key] = $data;
                } else {
                    $Intermediate_project[$key] = new TextFile($value);
                }
            }
        }
        catch (Exception $e) {
            echo "BlendedError while converting to Intermediary" . $e->getMessage();
        }
        
        return $Intermediate_project;
    }
    
    # Helper function to converts Intermediary list to dictionary and make json string to objects.
    function string_to_obj($intermediary, $jptf = array())
    {
        foreach ($intermediary as $key => $value) {
            if ($value instanceof DirectoryNode) {
                $jptf[$key] = $this->string_to_obj($value->data);
            } else if ($value instanceof BinaryFile) {
                $jptf[$key] = $value->location;
            } elseif ($value instanceof JsonFile) {
                $jptf[$key] = json_decode($value->data, true);
            } elseif($value instanceof TextFile) {
                $jptf[$key] = $value->data;
            }
        }
        
        return $jptf;
    }

    # Helper function to converts Intermediary list to dictionary with hashes and make json string to objects.
    function string_to_obj_hash($intermediary, $hash = null)
    {
        $jptf = array();
        if(!is_null($hash)) {
            $jptf['hash'] = $hash;
        }
        foreach ($intermediary as $key => $value) {
            if ($value instanceof DirectoryNode) {
                $jptf[$key]['hash'] = $value->hash;
                $jptf[$key]['data'] = $this->string_to_obj_hash($value->data);
            } else if ($value instanceof BinaryFile) {
                $jptf[$key] = $value->location;
            } elseif ($value instanceof JsonFile) {
                $jptf[$key] = json_decode($value->data, true);
            } elseif($value instanceof TextFile) {
                $jptf[$key] = $value->data;
            }
        }
        
        return $jptf;
    }
    
    # Transforming Intermediary project object to context format.
    function as_context($project, $trim = true)
    {
        
        $current_node      = null;
        $current_json_file = null;
        $dependency_dict   = array();
        $project           = $this->string_to_obj($project);
        
        try {
            if (isset($project["_project.json"])) {
                $project_json    = $project["_project.json"];
                $dependency_list = $project_json["dependencies"];
                
                # All dependencies withing project resolved here.
                if (!empty($dependency_list) && is_array($dependency_list)) {
                    $dependency_dict = $this->load_dependency($dependency_list);
                }
                
                # $ref resolution takes place here.
                $this->resolve($current_node, $project, $dependency_dict, $current_json_file);
                
                if ($trim == true) {
                    $project = $this->trim_context($project);
                }
            } else {
                # $ref resolution takes place here.                
                $this->resolve($current_node, $project, $dependency_dict, $current_json_file);
                if ($trim == true) {
                    $project = $this->trim_context($project);
                }
            }
            
            $project = $this->index_shortcuts($project);
        }
        catch (Exception $e) {
            throw new Exception("BlendedError while converting to Context");
        }
        
        return $project;
    }
    
    # Makes _index shortcuts available to project object.
    function index_shortcuts($project)
    {
        foreach ($project as $key => $value) {
            if (($key == "_index" || $key == "_index.json") && is_array($value)) {
                $project = array_merge($project, $value);
            } elseif (is_array($value)) {
                $this->index_shortcuts($value);
            }
        }
        return $project;
    }
    
    # Loads dependency in a project.
    function load_dependency($dependency_list)
    {
        global $ROOT_DIRECTORY;
        $dependency_dict = array();
        $project         = array();
        foreach ($dependency_list as $key => $value) {
            
            $dep_slug    = $value['name'];
            $dep_version = null;
            if (isset($value['version'])) {
                $dep_version = $value['version'];
            }
            
            $dep_alias    = $value['alias'];
            $dep_account  = $value['account'];
            $package_name = $dep_slug;
            
            # Call controller get_package to look for dependencies and return project object.
            $project                     = $this->get_package($package_name, 'context', null, $dep_version, null, false);
            $dependency_dict[$dep_alias] = $project;
        }
        
        return $dependency_dict;
    }
    
    # Removes all files starting with "_" and shortens resource paths.
    function trim_context($theme_object)
    {
        if (is_array($theme_object)) {
            foreach ($theme_object as $key => $value) {
                if ($key === "_index.json") {
                    $theme_object["_index.json"] = $this->trim_context($theme_object["_index.json"]);
                    $theme_object                = array_merge($theme_object, $theme_object["_index.json"]);
                    unset($theme_object["_index.json"]);
                } elseif (substr($key, 0, 1) == "_") {
                    unset($theme_object[$key]);
                } elseif (strpos($key, '.') !== false) {
                    $new_key                = explode('.', $key);
                    $new_key                = $new_key[0];
                    $theme_object[$new_key] = $value;
                    unset($theme_object[$key]);
                } else if (is_array($theme_object[$key])) {
                    $theme_object[$key] = $this->trim_context($theme_object[$key]);
                }
            }
        } elseif (is_string($theme_object)) {
            $theme_object = json_decode($theme_object, true);
        }
        
        return $theme_object;
    }
    
    # resolve all $ref within project.
    function resolve(&$current_node, &$project_object, $dependency_dict, &$current_json_file)
    {
        if (is_null($current_node)) {
            $current_node =& $project_object;
        }
        foreach ($current_node as $key => $value) {
            # Keep track of json files so that other objects can look here for refrence values.
            if (substr($key, -5) == ".json") {
                $current_json_file =& $current_node[$key];
            }
            $current_value =& $current_node[$key];
            # Look for the key '$ref' to resolve its pointing value.
            if (is_array($current_value)) {
                if (array_key_exists('$ref', $current_value)) {
                    $set = array();
                    try {
                        $current_node[$key] = $this->resolve_pointer($current_value, $project_object, $dependency_dict, $set, $current_json_file);
                    }
                    catch (Exception $e) {
                        $current_node[$key] = 'Caught exception: ' . $e->getMessage() . "\n";
                    }
                } else {
                    $this->resolve($current_value, $project_object, $dependency_dict, $current_json_file);
                }
            }
        }
    }
    
    
    # finds occurences for any singular "$ref", find value and fix. A helper function for resolve.
    function &resolve_pointer(&$json_ref, &$project_object, $dependency_dict, $visited_set, &$current_json_file)
    {
        if (is_array($json_ref) && array_key_exists('$ref', $json_ref)) {
            $pointer = $json_ref['$ref'];
            
            if (in_array($pointer, $visited_set)) {
                # Entered into cycle.
                throw new Exception('A Json Pointer cycle has been detected');
            } else {
                # Push current item to stack for backtrakking cycles in path.
                array_push($visited_set, $pointer);
                
                $ref_value = $pointer;
                $json_file = $current_json_file;
                
                # Look further to resolve local/external dependencies.
                if (strpos($ref_value, "#/") === 0) {
                    $path      = str_replace('#/', '', $ref_value);
                    $ref_value = $this->get_value_at_path($path, $json_file);
                } elseif (strpos($ref_value, "/") === 0) {
                    $ref_value = ltrim($ref_value, "/");
                    $path      = str_replace('#', '', $ref_value);
                    $ref_value = $this->get_value_at_path($path, $project_object);
                } elseif (strpos($ref_value, "@") === 0) {
                    $path      = str_replace(array(
                        '@',
                        '#'
                    ), '', $ref_value);
                    $ref_value = $this->get_value_at_path($path, $dependency_dict);
                } else {
                    $path      = str_replace('#', '', $ref_value);
                    $ref_value = $this->get_value_at_path($path, $project_object);
                }
                $json_ref = $ref_value;
                if (empty($json_ref)) {
                    throw new Exception('value not found');
                }
                
                $this->resolve_pointer($json_ref, $project_object, $dependency_dict, $visited_set, $current_json_file);
            }
        }
        return $json_ref;
    }
    
    # Detects for json pointer($ref) path for local or theme references. A helper function for resolve.
    function &get_value_at_path($dir_path, &$object)
    {
        $paths = explode('/', $dir_path);
        $temp_value =& $object;
        foreach ($paths as $key => $value) {
            if (count($temp_value) == 1 & isset($temp_value['$ref'])) {
                $json_ref          = $temp_value;
                $dependency_dict   = array();
                $visited_set       = array();
                $current_json_file = $object;
                $temp_value        = $this->resolve_pointer($json_ref, $object, $dependency_dict, $visited_set, $current_json_file);
            }
            $temp_value =& $temp_value[$value];
        }
        return $temp_value;
    }
    
    # Converts Intermediary project object to jptf format.
    function as_jptf($Intermediary_object, $flag = true)
    {
            $jptf_object = array();
            $Intermediary_data = $Intermediary_object->data;

            $jptf_object = $this->as_jptf_builder($Intermediary_data);
            $jptf_object = $this->JptfList($jptf_object);

            $jptf[0]['/']['hash'] = $Intermediary_object->hash;
            $jptf[0]['/']['type'] = "directory";
            $jptf[0]['/']['content'] = $jptf_object;
        return $jptf;
    }

    function as_jptf_builder($Intermediary_object) {

          $jptf_object = array();

        try {
            foreach ($Intermediary_object as $key => $value) {
                if(is_int($key)) {
                    $jptf_object[$key] = $this->as_jptf_builder($value);
                }
                elseif ($value instanceof DirectoryNode) {
                    $jptf_object[$key]["type"]     = "directory";
                    $jptf_object[$key]["hash"]     = $value->hash;
                    $jptf_object[$key]["content"] = $this->as_jptf_builder($this->JptfList($value->data));
                } elseif ($value instanceof TextFile) {
                    //$errors = array_filter($value->data);
                    //if(isset$value->data)
                    $jptf_object[$key]["hash"]    = md5($value->data);
                    $jptf_object[$key]["type"]    = "file";
                    $jptf_object[$key]["content"] = base64_encode(gzcompress($value->data, 6));
                } elseif($value instanceof JsonFile) {
                    $jptf_object[$key]["type"]    = "file";
                    $jptf_object[$key]["hash"]    = md5($value->data);
                    $jptf_object[$key]["content"] = base64_encode(gzcompress(json_encode($value->data), 6));
                } elseif($value instanceof BinaryFile) {
                    $jptf_object[$key]["hash"]    = $value->hash;
                    $jptf_object[$key]["type"]    = "media";
                }
            }
        }
        catch (Exception $e) {
            throw new Exception("BlendedError while converting to jptf");
        }
        return $jptf_object;
    }

    # update _package.json
    function updatePackageJson($name, $value, $package_slug) {
        $content = file_get_contents($package_slug . '/_project.json');
        $content = json_decode($content, true);

        if(array_key_exists($name, $content)) {
           $content[$name] = $value;
           $content = json_encode($content);
           file_put_contents($package_slug . '/_project.json', $content);
           return true;
        }

        return false;
    }

    # Get file jptf
    function getFile($file_path, $package_slug) {

        $backend = new Backend($package_slug);
        $project = $backend->get_package(); 
        $file_path = array($file_path);
        $file_jptf = $this->createPartialJptf($file_path, $project);
        return $file_jptf;
    }
    
    # creates a partial jptf package to push.
    function createPartialJptf($list, $Intermediary_object)
    {
        $object_dictionary  = $this->string_to_obj_hash($Intermediary_object->data, $Intermediary_object->hash);
        $partial_dict = array();
        foreach ($list as $key => $value) {
            $iterate_path = explode('/', $value);
            $file_name    = basename($value);
            $file_data    = $this->find_fileData($iterate_path, $object_dictionary);
            $data = $file_data['content'];
            $directory_hashes = $file_data['hash_list'];

            $partial_dict = array_merge_recursive($this->createPartialDict($iterate_path, $file_name, $data, $directory_hashes), $partial_dict);

        }

        $partialJptfData = $this->builtJptf($this->JptfList($partial_dict));
        $partialJptf[0]['/']['type'] = "directory";
        $partialJptf[0]['/']['hash'] = $Intermediary_object->hash;
        $partialJptf[0]['/']['content'] = array();

        foreach($partialJptfData as $key => $value) {
           $partialJptf[0]['/']['content'] = array_merge($partialJptfData[$key], $partialJptf[0]['/']['content']);
        }
          $unsortedJptf = $this->JptfList($partialJptf[0]['/']['content']);
          sort($unsortedJptf);
          $partialJptf[0]['/']['content'] = $unsortedJptf;

        return $partialJptf;
    }

    function createPartialDict($iterate_path, $file_name, $file_data, $directory_hashes) {
         $data = array();
         foreach($iterate_path as $key => $value) {
            if($value != $file_name) {              
               $data[$value] = array_merge($this->createPartialDict($this->unset_jptfIterator($key, $iterate_path), $file_name, $file_data, $directory_hashes), $data);
               $data[$value]['hash'] = $directory_hashes[$value];
               break;
            } else {
               $data[$value] = $file_data;               
            }
         }
          return $data;
    }
    
    # Get file data in the $object_dictionary.
    function find_fileData($iterate_path, $object_dictionary)
    {
        $temp = $object_dictionary;
        $directory_hash_list = array();
        foreach ($iterate_path as $key => $value) {
            if(strpos($value, '.') !== true && isset($temp[$value]['data'])) {
                 $directory_hash_list[$value] = & $temp[$value]['hash'] ;
                 $temp =& $temp[$value]['data'];
            } else {
                 $temp =& $temp[$value];
            }

        }
        $hash_list['hash_list'] = $directory_hash_list;
        $temp = array_merge($hash_list, array('content' =>$temp));

        return $temp;
    }
    
    # Create file jptf and insert file data.
    function builtJptf($partial_dict)
    {

        global $allowed_image_types;
        $partial_jptf = array();
        $count = 0;
        foreach($partial_dict as $key => $value) {

           $file_extension = explode('.', $key);
           $file_extension = $file_extension[count($file_extension) - 1];

           if(is_int($key)) {
                $partial_jptf[$key] = $this->builtJptf($value);
           }
           elseif(substr($key, -5) == ".json") {
                $value = json_encode($value);
                $partial_jptf[$key]['type'] = "file";
                $partial_jptf[$key]['content'] = base64_encode(gzcompress($value, 6));
                $partial_jptf[$key]['hash'] = md5($value);           
           } elseif(strpos($key, '.') !== true && is_array($value)) {
                $partial_jptf[$key]['type'] = "directory";
                $partial_jptf[$key]['hash'] = $value['hash'];
                unset($value['hash']);
                $partial_jptf[$key]['content'] = $this->builtJptf($this->JptfList($value));
                $count++;
           } else if(in_array(strtolower($file_extension), $allowed_image_types)) {
                $partial_jptf[$key]['type'] = "media";
                $partial_jptf[$key]['hash'] = md5($value);
           } elseif(strpos($key, '.') !== false) {
                $partial_jptf[$key]['type'] = "file";
                $partial_jptf[$key]['content'] = base64_encode(gzcompress($value, 6));
                $partial_jptf[$key]['hash'] = md5($value);
           }
        }

        return $partial_jptf;
    }

    function JptfList($dict) {
       $list = array();
       foreach($dict as $key => $value) {
           $list[][$key] = $value;
       }
       return $list;
    }
    
    # unset visited path from path iterator.
    function unset_jptfIterator($key, $array)
    {
        unset($array[$key]);
        return $array;
    }
}

class IntermediaryNode
{
    private $hash;
    var $data;
    var $location;
    function __construct($location, $package = null, $hash = null)
    {
        $this->data     = $package;
        $this->location = basename($location);
        $this->hash     = $hash;
    }
    
    function getData($file_object)
    {
        if (file_exists($file_object)) {
            return file_get_contents($file_object);
        } else {
            return $file_object;
        }
    }
    
    function getHash($packageSlug)
    {
        if (file_exists($packageSlug)) {
            $package      = file_get_contents($packageSlug . '/.hash');
            $packageHash = json_decode($package, true);

            if (isset($packageHash['packageHash']) && !empty($packageHash['packageHash'])) {
                return $packageHash['packageHash'];
            }
        } else {
            // call to generateHash
        }
    }
    
    function get_package($dir)
    {
        global $ACTIVE_THEME;
        global $allowed_image_types;
        
        $dir = new DirectoryIterator($dir);
        $theme_object = array();
        # Iterating local theme directory to form Intermediary theme object.
        foreach ($dir as $key => $node) {
           // print_r($node->getFilename());echo"\n";
            if ($node->isDir() && !$node->isDot() && substr($node->getFilename(), 0, 1) !== "_") {
                $theme_object[$node->getFilename()] = new DirectoryNode($node->getPathname());
            } else if ($node->isFile() && $node->getFilename() == "_index.json" || $node->isFile() && substr($node->getFilename(), 0, 1) !== "_" && substr($node->getFilename(), -1) !== "~" || $node->isFile() && $node->getFilename() == "_project.json") {
                
                # Getting extension of a file.
                $file_extension = explode('.', $node->getFilename());
                $file_extension = $file_extension[count($file_extension) - 1];
                
                # Checking json or media type file else load content for all other.
                if (substr($node->getFilename(), -5) == ".json") {
                    $file_location                      = $node->getpath() . "/" . $node->getFilename();
                    $file_contents                      = new JsonFile($file_location);
                    $theme_object[$node->getFilename()] = $file_contents;
                } else if (in_array(strtolower($file_extension), $allowed_image_types)) {
                    # If file name matches with any of the allowed binary extension then store "<theme_name>/path/to/media"   
                    $file_location                      = $node->getpath() . "/" . $node->getFilename();
                    //$theme_dir                          = basename($ACTIVE_THEME);
                    //$path_track                         = $node->getpath() . "/" . $node->getFilename();
                    //$path_track                         = str_replace('\\', '/', $path_track);
                    //$path_track                         = explode($theme_dir . '/', $path_track);
                    //$binary_path                        = $path_track[count($path_track) - 1];
                    $theme_object[$node->getFilename()] = new BinaryFile($file_location);
                } else if($node->getFilename() != ".hash.txt" && $node->getFilename() != ".hash" && substr($node->getFilename(), -1) !== "~") {
                    # For all other type files, get file contents.
                    $theme_object[$node->getFilename()] = new TextFile($node->getpath() . "/" . $node->getFilename());
                }
            }
        }
        
        return $theme_object;
    }
}

class DirectoryNode extends IntermediaryNode
{
    
    var $hash;
    function __construct($dir, $package = null)
    {
        if (!file_exists($dir)) {
            parent::__construct($dir, $package);
        } else {
            $package = $this->get_package($dir);
            parent::__construct($dir, $package);
            $this->hash = $this->generateHash();
        }
    }
    
    
    
    function generateHash()
    {
        global $ACTIVE_THEME;        
        
        foreach ($this->data as $key => $value) {
            if ($key != ".blended" && $key != "blendedrc") {
                $package_hash_list[] = basename($ACTIVE_THEME) . ':' . $value->hash;
            }
            
        }
        
        if (isset($package_hash_list)) {
            sort($package_hash_list);
            $package_hash_string = implode('::', $package_hash_list);
            $package_hash        = $package_hash_string;
            $this->hash          = md5($package_hash);
            return $this->hash;
        }
    }
}

class TextFile extends IntermediaryNode
{
    function __construct($file_location)
    {
        $this->data     = $this->getData($file_location);
        $this->location = basename($file_location);
        $this->hash     = $this->generateHash();
    }
    
    function generateHash()
    {
        return md5($this->data);
    }
}

class JsonFile extends IntermediaryNode
{
    function __construct($file_location)
    {
        $this->data     = $this->getData($file_location);
        $this->location = basename($file_location);
        $this->hash     = $this->generateHash();
    }
    
    function generateHash()
    {
        return md5($this->data);
    }
}

class BinaryFile extends IntermediaryNode
{
    
    function __construct($file_location)
    {
        $file_path      = $file_location;
        $this->hash     = $this->generateHash($file_path);
        $this->location = basename($file_path);
    }
    
    function generateHash($file_path)
    {
        if(file_exists($file_path)) 
        {
           return sha1_file($file_path);
        }
       return $file_path;
    }
    
    function _calculateHash()
    {
    }
}

?>
