<?php

/**
 * Blended Navigation Functions
 * @since 1.0
 */

/**
 * Retrieve nav menu.
 * @param string $menu
 * @param array $args
 * @return array
 */
function blended_nav_menu($menu = null, $args = array()) {

    $defaults = array('width_nv' => 0, 'length_nv' => 0, 'depth_nv' => 0,);
    $args = wp_parse_args($args, $defaults);

    /**
     * Filters the arguments used to display a navigation menu.
     *
     * @since 1.0.0
     *
     * @see blended_nav_menu()
     *
     * @param array $args Array of blended_nav_menu() arguments.
     */
    $args = apply_filters('blended_nav_menu_args', $args);
    $args = (object) $args;

    /**
     * Get navigation settings
     */
    $backend = new Backend();
    $network = new Network();
    $controller = new Controller($backend, $network);
    $menu_settings = $controller->get_navigation_settings();

    if ($menu_settings && array_key_exists($menu, $menu_settings)):

        //Set default menu, if no types
        if (!isset($menu_settings[$menu]["types"])) :
            $menu = "";
        else:
            switch ($menu_settings[$menu]["types"]) {
                case "host":

                    $menu = $menu_settings[$menu]["id"];
                    break;

                case "url":

                    $response = wp_remote_get($menu_settings[$menu]["data"], array(
                        'timeout' => 120));

                    if (is_array($response)) {
                        return json_decode($response['body']);
                    }
                    return;

                    break;

                case "custom":

                    return json_decode($menu_settings[$menu]['data'], true);

                    break;
            }

        endif;

    endif;

    //TimberMenu
    $menu_object = new TimberMenu($menu);

    $menu_items = array();

    //Menu width flag
    $menu_width_flag = 1;

    if ($menu_object):
        //Get menu item for parent node
        $parent_items = array();
        foreach ($menu_object->get_items as $parent_item) :
            $parent_items["label"] = $parent_item->title;
            $parent_items["href"] = $parent_item->get_link;
            $parent_items["classes"] = array_filter($parent_item->classes);
            $parent_items["class"] = $parent_item->class;
            $parent_items["id"] = $parent_item->id;
            $parent_items["selected"] = in_array('current-menu-item', $parent_item->classes) ? true : false;


            //Set child menu items
            $parent_items["children"] = blended_nav_child_items($parent_item, $args->length_nv, $args->depth_nv);

            $menu_items[] = $parent_items;


            //Width flag
            if ($menu_width_flag === $args->width_nv)
                break;
            $menu_width_flag++;

        endforeach;

        //Prepare menu object
        $menu_data = array(
            "id" => $menu_object->id,
            "name" => $menu_object->name,
            "label" => $menu_object->slug,
            "nav_items" => $menu_items
        );

        //Reset the postdata
        wp_reset_postdata();

        //We are done!
        return $menu_data;
    endif;
}

/**
 * Retrieve nav menu child items.
 *
 * @param object parent menu item
 * @return array child items.
 */
function blended_nav_child_items($parent_menu_item, $length, $depth, $flag = 1) {
    $menu_child_items = array();

    /**
     * @var Length flag
     */
    $menu_length_flag = 1;

    if ($flag === $depth)
        return $menu_child_items;

    if ($parent_menu_item->get_children):
        $child_items = array();
        foreach ($parent_menu_item->get_children as $child_item):

            $flag++;

            $child_items["label"] = $child_item->title;
            $child_items["href"] = $child_item->get_link;
            $child_items["classes"] = array_filter($child_item->classes);
            $child_items["class"] = $child_item->class;
            $child_items["id"] = $child_item->id;
            $child_items["selected"] = in_array('current-menu-item', $child_item->classes) ? true : false;

            /**
             * Get nested child items using recursion
             */
            $child_items["children"] = blended_nav_child_items($child_item, $length, $depth, $flag);
            $menu_child_items[] = $child_items;


            //Length flag
            if ($menu_length_flag === $length)
                break;
            $menu_length_flag++;

        endforeach;

    endif;

    return $menu_child_items;
}
